
/*
 * $Header: /cvsroot/mpdist/mpdist/mp/input.c,v 1.1.1.1 2002/04/12 16:47:25 richbastard Exp $
 * 
 * Copyright (c) 1987-2002 Rich Burridge, Sun Microsystems Inc.
 * All rights reserved.
 * 
 * This software is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include "option.h"
#include "header.h"
#include "input.h"
#include "print.h"

char *owner;                      /* Name of owner (usually equal to 'to') */

static FILE *ifp;                 /* Input file pointer for current file. */

static char iobuf[MAXLINE];    /* Overflow buffer when -wrap is used. */
static char *nameptr;          /* Used to getenv the NAME variable. */
static char nextline[MAXLINE]; /* Read-ahead of mail message, minus nl. */
static char *whoami;           /* Login name of user. */

static document_type dtype;    /* Printing type - default mail. */

static int clen;               /* Current line length (including newline). */
static int eof;                /* End of file indicator. */
static int eol;                /* End of line indicator. */
static int eop;                /* End of page indicator. */
static int linenum;            /* Current line number from input file. */
static int mlen;        /* Number of characters in message (-C option). */
static int overflow;    /* # of chars of text wraparound if -wrap. */

static void process_name_field (char *, char *, int, int);


void
close_input(void)       /* Close current input channel. */
{
    FCLOSE(ifp);
}


char *
current_line(void)      /* Return current input line. */
{
    return(nextline);
}


document_type
doc_type(void)
{
    return(dtype);
}


int
end_of_file(void)
{
    return(eof);
}


int
end_of_line(void)
{
    return(eol);
}


int
end_of_page(void)
{
    return(eop);
}


char *
get_line()
{
    return(fgets(nextline, MAXLINE, ifp));
}


void
init_setup(void)             /* Set default values for various options. */
{
    char *aliasname, *c;
    int amp_cnt = 0;         /* Number of ampersands in gecos field. */
    int i, len;
    struct passwd *pp;

    c = getlogin();          /* Pointer to users login name. */
    if (c == NULL || *c == '\0') {    /* Get username from password file. */
        pp = getpwuid(geteuid());
        if (pp == NULL) {
            c = "printing";
        } else {
            c = pp->pw_name;
        }
    }
    aliasname = str_opt_val(O_ALIAS);
    if (aliasname && *aliasname != '\0') {
        c = aliasname;        /* Use alias if defined. */
    }
    owner = (char *) malloc((unsigned) (strlen(c) + 1));
    STRCPY(owner, c);
    whoami = (char *) malloc((unsigned) (strlen(c) + 1));
    STRCPY(whoami, c);        /* Save User login name. */

/*
 * Have a look for the users gecos (normally real name), so that its
 * a bit more recognisable. If this field is too long, then we need
 * to truncate sensibly. We also need to check a few things. If we've
 * extracted namefields "words" or have found a comma, then exit. If
 * an ampersand is found, this is expanded to the users name in
 * capitals.
 */

    pp = getpwnam(owner);
    if (pp != NULL && pp->pw_gecos && pp->pw_gecos[0] != '\0') {
        len = strlen(pp->pw_gecos);
        for (i = 0; i < len; i++) {
            if (pp->pw_gecos[i] == '&') {
                amp_cnt++;
            }
        }

        nameptr = getenv("NAME");
        if (nameptr != NULL) {
            len = strlen(nameptr);
        } else {
            len = strlen(pp->pw_gecos) + amp_cnt * strlen(c);
        }

        owner = (char *) malloc((unsigned) (len + 1));

        if (nameptr != NULL) {
            process_name_field(c, nameptr, int_opt_val(O_NAMEF),
                       int_opt_val(O_NAMELEN));
        } else {
            process_name_field(c, pp->pw_gecos, int_opt_val(O_NAMEF),
                       int_opt_val(O_NAMELEN));
        }
    }
    if (int_opt_val(O_TEXTDOC)) {
        dtype = DO_TEXT;
    }
    switch (dtype) {
        case DO_TEXT:
            set_message_for("Listing for ");
            set_int_opt_val(O_DIGEST, 0);
            break;
        case DO_MAIL:
            set_message_for(int_opt_val(O_DIGEST) ? "Mail digest for " 
                                                  : "Mail for ");
            break;
        case DO_NEWS:
            set_message_for(int_opt_val(O_DIGEST) ? "News digest for " 
                                                  : "News for ");
            break;
    }
}


void
Input(void)
{
    eof = 0;             /* EOF indicator. */
    linenum = 0;         /* Current line number from input file. */
    overflow = 0;        /* Initially no text wraparound. */
    dtype = DO_MAIL;     /* Printing type - default mail. */
}


int
line_len(void)           /* Return the current line length. */
{
    return(clen);
}


int
message_len(void)        /* Return the current message length. */
{
    return(mlen);
}


/* Extract user name from $NAME or passwd GECOS. */

static void
process_name_field(char *c, char *ptr, int fields, int length)
{
    int i, j, len, n, spaces, slen;

    n = spaces = 0;
    slen = strlen(ptr);
    for (i = 0; i < slen; i++) {
        if (*ptr == ',') {
            break;
        } else if (*ptr == '&') {
            if (islower((int) c[0])) {
                owner[n++] = toupper(c[0]);
            }
            len = strlen(c);
            for (j = 1; j < len; j++) {
                owner[n++] = c[j];
            }
            ptr++;
        } else if (*ptr == ' ' || *ptr == '\t') {
            if (++spaces == fields) {
                break;
            } else {
                while (*ptr == ' ' || *ptr == '\t') {
                    owner[n++] = *ptr++;
                }
            }
        } else {
            owner[n++] = *ptr++;
        }
        if (n >= length) {
            break;
        }
    }
    if (n > length) {
        n = length;
    }
    owner[n] = '\0';
}


/* Read an input line into nextline, setting eof, eop and eol appropriately. */

void
readline(void)
{
    char *ptr;
    int llen;
    int c = 0;
    int i = 0;             /* Index into current line being read. */
    int len = 0;           /* Length of the current line. */
    int startpos = 0;      /* Start position of real text in line. */

    if (eof) {
        return;
    }
    llen = int_opt_val(O_LLEN);
    eop = eol = 0;

    if (int_opt_val(O_NUMBER)) {    /* Should output lines be numbered? */
        SPRINTF(nextline, "%6d  ", ++linenum);
        startpos = len = i = 8;
    }
    if (overflow) {                 /* Is this -wrap with word wraparound? */
        STRNCPY(&nextline[len], iobuf, overflow);
        len += overflow;
        i += overflow;
        overflow = 0;
    }
    while (len < llen && (c = getc(ifp)) != EOF && c != '\n' && c != '\f') {
        if (c == '\t') {
            do {
                nextline[i++] = ' ';
                len++;
            } while (len % int_opt_val(O_TABSIZE) != 0 && len <= llen);
        } else {
            nextline[i++] = c;
            len++;
        }
        if (c == '\b') {
            len -= 2;
            i -= 2;
        }
    }
    nextline[i] = '\0';

    if (len == llen && c != EOF && c != '\n' && c != '\f') {
        c = getc(ifp);
        if (c != EOF && c != '\n' && c != '\f') {
            UNGETC(c, ifp);
        }

        if (int_opt_val(O_WRAPW) && (ptr = strrchr(nextline, ' ')) != NULL) {
            STRCPY(iobuf, ptr + 1);
            len = ptr - nextline + 1;
            nextline[len] = '\0';     /* Terminate current line. */
            overflow = llen - len;    /* # of chars wrapped. */
        }
    }
    if (int_opt_val(O_ELM) && c == '\f') {
        len--;
        i--;
    }
    switch (c) {
        case EOF:
            if (i == startpos) {
                eof = 1;
            } else {
                UNGETC(c, ifp);
                eol = 1;
            }
            break;

        case '\n':
            eol = 1;
            break;

/*
 * /usr/ucb/mail for some unknown reason, appends a bogus formfeed at the 
 * end of piped output. The next character is checked; if it's an EOF, 
 * then eof is set, else the character is put back.
 */

        case '\f':
            if ((c = getc(ifp)) == EOF) {
                eof = 1;
            } else {
                UNGETC(c, ifp);
            }

            eol = 1;
            eop = 1;
            break;
    }

    clen = len + 1;        /* Current line length (includes newline). */
}


void
set_doc_type(document_type dt)
{
    dtype = dt;
}


void
set_eof(int val)
{
    eof = val;
}


void
set_eop(int val)
{
    eop = val;
}


void
set_input(FILE * fp)
{
    ifp = fp;
}


void
set_line_number(int val)
{
    linenum = val;
}


void
set_message_len(int len)        /* Set the current message length. */
{
    mlen = len;
}


char *
who_am_i(void)
{
    return(whoami);
}
