/*
    fungw - function gateway
    Copyright (C) 2017 Tibor 'Igor2' Palinkas

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA

    Project page: http://repo.hu/projects/fungw
    Version control: svn://repo.hu/fungw/trunk
*/

/* These functions can be called from an scconfig hooks.c to get libfungw
   bindings configured without having to have or compile or run a separate
   scconfig process for the lib. Assumes libfungw/scconfig_hooks.h functions
   has been already called. */

#include <stdio.h>
#include <string.h>
#include "arg.h"
#include "log.h"
#include "dep.h"
#include "db.h"
#include "tmpasm_scconfig.h"

/* returns if a language is enabled (not disabled) */
#define EN(name)  (!istrue(get("/local/fungwbind/disable-" name)))

static int fungwbind_hook_postinit(void)
{
	db_mkdir("/local");
	put("/local/fungw/bindings_all", "c fawk lua mawk tcl picol python python3 mruby perl duktape mujs estutter funlisp cli");
/* TODO: welltype should be on the above list */
	put("/local/fungw/bindings", "");
	return 0;
}

static int fungwbind_hook_detect_target(void)
{
	int static_disable_python = 0;
	if (EN("fawk")) put("libs/script/fawk/presents", strue);
	put("libs/script/c/presents", strue); /* can not be disabled */
	if (EN("lua")) require("libs/script/lua/*", 0, 0);
	if (EN("mawk")) require("libs/script/mawk/*", 0, 0);
	if (EN("tcl")) require("libs/script/tcl/*", 0, 0);
	if (EN("python")) {
		if (require("libs/script/python/*", 0, 0) == 0)
			static_disable_python = 3;
	}
	if (EN("python3")) {
		if (require("libs/script/python3/*", 0, 0) == 0)
			static_disable_python = 2;
	}
	if (EN("mruby")) require("libs/script/mruby/*", 0, 0);
	if (EN("perl")) require("libs/script/perl", 0, 0);
	if (EN("duktape")) require("libs/script/duktape/*", 0, 0);
	if (EN("estutter")) require("libs/script/estutter/*", 0, 0);
	if (EN("funlisp")) require("libs/script/funlisp/*", 0, 0);
	if (EN("picol")) require("libs/script/picol/*", 0, 0);
#if 0
	if (EN("welltype")) {
		if (require("libs/script/welltype/wtc/*", 0, 0) == 0)
			require("libs/script/welltype/*", 0, 0);
	}
#endif

	if (EN("mujs")) {
		require("libs/script/mujs/*", 0, 0);
		require("libs/script/mujs_isboolean/*", 0, 0);
		if (!istrue(get("libs/script/mujs_isboolean/presents")) && istrue(get("libs/script/mujs/presents"))) {
			report("WARNING: Your mujs is probably too old: it doesn't have JS_ISBOOLAN - disabling mujs\n");
			put("libs/script/mujs/presents", sfalse);
		}
	}

	/* for cli */
	if (EN("cli")) {
		require("libs/proc/fork/*", 0, 0);
		require("libs/io/pipe/*", 0, 0);
	}

	if (static_disable_python != 0) {
		char path[64];
		if (static_disable_python == 2)
			strcpy(path, "/local/fungw/binding/static-disable/python");
		else
			sprintf(path, "/local/fungw/binding/static-disable/python%d", static_disable_python);
		put(path, strue);
	}

	return 0;
}

#define gen_lang(langdir) \
	fprintf(stderr, "Generating libfungwbind/" langdir "/Makefile (%d)\n", generr |= tmpasm(root, langdir "/Plug.tmpasm", langdir "/Makefile"));

/* Root is the path for the fungwlug lib dir (trunk/libfungwbind) */
static int fungwbind_hook_generate(const char *root)
{
	int generr = 0;

	gen_lang("c");
	gen_lang("fawk");
	gen_lang("lua");
	gen_lang("mawk");
	gen_lang("tcl");
	gen_lang("picol");
	gen_lang("python");
	gen_lang("python3");
	gen_lang("duktape");
	gen_lang("mujs");
	gen_lang("mruby");
	gen_lang("perl");
	gen_lang("estutter");
	gen_lang("funlisp");
#if 0
	gen_lang("welltype");
#endif
	gen_lang("cli");

	fprintf(stderr, "Generating libfungwbind/Makefile (%d)\n", generr |= tmpasm(root, "Makefile.in", "Makefile"));
	fprintf(stderr, "Generating libfungwbind/libfungwbind.mak (%d)\n", generr |= tmpasm(root, "libfungwbind.mak.in", "libfungwbind.mak"));

	return generr;
}

static void fungwbind_help1(void)
{
	printf(" --disable-fawk         do not compile fawk support\n");
	printf(" --disable-lua          do not compile lua support\n");
	printf(" --disable-mawk         do not compile mawk support\n");
	printf(" --disable-tcl          do not compile tcl support\n");
	printf(" --disable-picol        do not compile picol support\n");
	printf(" --disable-python       do not compile python2 support\n");
	printf(" --disable-python3      do not compile python3 support\n");
	printf(" --disable-duktape      do not compile duktape (js) support\n");
	printf(" --disable-mujs         do not compile mujs (js) support\n");
	printf(" --disable-mruby        do not compile mruby support\n");
	printf(" --disable-perl         do not compile perl support\n");
	printf(" --disable-estutter     do not compile estutter support\n");
	printf(" --disable-funlisp      do not compile funlisp support\n");
#if 0
	printf(" --disable-welltype     do not compile welltype support\n");
#endif
	printf(" --disable-cli          do not compile cli support\n");
}

#define CUSTOM_ARG(name) \
	if (strcmp(key, name) == 0) { \
		put("/local/fungwbind/disable-" name, strue); \
		return 1; \
	}

static int fungwbind_custom_arg(const char *key, const char *value)
{
	if (strncmp(key, "disable-", 8) != 0)
		return 0;

	key += 8;

	CUSTOM_ARG("fawk");
	CUSTOM_ARG("lua");
	CUSTOM_ARG("mawk");
	CUSTOM_ARG("tcl");
	CUSTOM_ARG("picol");
	CUSTOM_ARG("python");
	CUSTOM_ARG("python3");
	CUSTOM_ARG("duktape");
	CUSTOM_ARG("mujs");
	CUSTOM_ARG("mruby");
	CUSTOM_ARG("perl");
	CUSTOM_ARG("estutter");
	CUSTOM_ARG("funlisp");
#if 0
	CUSTOM_ARG("welltype");
#endif
	CUSTOM_ARG("cli");

	return 0;
}
