#ifndef CAPSTONE_SYSTEMZ_H
#define CAPSTONE_SYSTEMZ_H

/* Capstone Disassembly Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2014-2015 */
/* By Rot127 <unisono@quyllur.org>, 2024 */

#ifdef __cplusplus
extern "C" {
#endif

#include "platform.h"
#include "cs_operand.h"

/// Enums corresponding to SystemZ condition codes
typedef enum systemz_cc {
	SYSTEMZ_CC_O,
	SYSTEMZ_CC_H,
	SYSTEMZ_CC_NLE,
	SYSTEMZ_CC_L,
	SYSTEMZ_CC_NHE,
	SYSTEMZ_CC_LH,
	SYSTEMZ_CC_NE,
	SYSTEMZ_CC_E,
	SYSTEMZ_CC_NLH,
	SYSTEMZ_CC_HE,
	SYSTEMZ_CC_NL,
	SYSTEMZ_CC_LE,
	SYSTEMZ_CC_NH,
	SYSTEMZ_CC_NO,
	SYSTEMZ_CC_INVALID,
} systemz_cc;

/// Operand type for instruction's operands
typedef enum systemz_op_type {
	/// CS_OP_INVALID (Uninitialized).
	SYSTEMZ_OP_INVALID = CS_OP_INVALID,
	SYSTEMZ_OP_REG = CS_OP_REG, ///< = CS_OP_REG (Register operand).
	SYSTEMZ_OP_IMM = CS_OP_IMM, ///< = CS_OP_IMM (Immediate operand).
	SYSTEMZ_OP_MEM = CS_OP_MEM, ///< = CS_OP_MEM (Memory operand).
} systemz_op_type;

/// SystemZ registers
typedef enum systemz_reg {
	// generated content <SystemZGenCSRegEnum.inc> begin
	// clang-format off

	SYSTEMZ_REG_INVALID = 0,
	SYSTEMZ_REG_CC = 1,
	SYSTEMZ_REG_FPC = 2,
	SYSTEMZ_REG_A0 = 3,
	SYSTEMZ_REG_A1 = 4,
	SYSTEMZ_REG_A2 = 5,
	SYSTEMZ_REG_A3 = 6,
	SYSTEMZ_REG_A4 = 7,
	SYSTEMZ_REG_A5 = 8,
	SYSTEMZ_REG_A6 = 9,
	SYSTEMZ_REG_A7 = 10,
	SYSTEMZ_REG_A8 = 11,
	SYSTEMZ_REG_A9 = 12,
	SYSTEMZ_REG_A10 = 13,
	SYSTEMZ_REG_A11 = 14,
	SYSTEMZ_REG_A12 = 15,
	SYSTEMZ_REG_A13 = 16,
	SYSTEMZ_REG_A14 = 17,
	SYSTEMZ_REG_A15 = 18,
	SYSTEMZ_REG_C0 = 19,
	SYSTEMZ_REG_C1 = 20,
	SYSTEMZ_REG_C2 = 21,
	SYSTEMZ_REG_C3 = 22,
	SYSTEMZ_REG_C4 = 23,
	SYSTEMZ_REG_C5 = 24,
	SYSTEMZ_REG_C6 = 25,
	SYSTEMZ_REG_C7 = 26,
	SYSTEMZ_REG_C8 = 27,
	SYSTEMZ_REG_C9 = 28,
	SYSTEMZ_REG_C10 = 29,
	SYSTEMZ_REG_C11 = 30,
	SYSTEMZ_REG_C12 = 31,
	SYSTEMZ_REG_C13 = 32,
	SYSTEMZ_REG_C14 = 33,
	SYSTEMZ_REG_C15 = 34,
	SYSTEMZ_REG_V0 = 35,
	SYSTEMZ_REG_V1 = 36,
	SYSTEMZ_REG_V2 = 37,
	SYSTEMZ_REG_V3 = 38,
	SYSTEMZ_REG_V4 = 39,
	SYSTEMZ_REG_V5 = 40,
	SYSTEMZ_REG_V6 = 41,
	SYSTEMZ_REG_V7 = 42,
	SYSTEMZ_REG_V8 = 43,
	SYSTEMZ_REG_V9 = 44,
	SYSTEMZ_REG_V10 = 45,
	SYSTEMZ_REG_V11 = 46,
	SYSTEMZ_REG_V12 = 47,
	SYSTEMZ_REG_V13 = 48,
	SYSTEMZ_REG_V14 = 49,
	SYSTEMZ_REG_V15 = 50,
	SYSTEMZ_REG_V16 = 51,
	SYSTEMZ_REG_V17 = 52,
	SYSTEMZ_REG_V18 = 53,
	SYSTEMZ_REG_V19 = 54,
	SYSTEMZ_REG_V20 = 55,
	SYSTEMZ_REG_V21 = 56,
	SYSTEMZ_REG_V22 = 57,
	SYSTEMZ_REG_V23 = 58,
	SYSTEMZ_REG_V24 = 59,
	SYSTEMZ_REG_V25 = 60,
	SYSTEMZ_REG_V26 = 61,
	SYSTEMZ_REG_V27 = 62,
	SYSTEMZ_REG_V28 = 63,
	SYSTEMZ_REG_V29 = 64,
	SYSTEMZ_REG_V30 = 65,
	SYSTEMZ_REG_V31 = 66,
	SYSTEMZ_REG_F0D = 67,
	SYSTEMZ_REG_F1D = 68,
	SYSTEMZ_REG_F2D = 69,
	SYSTEMZ_REG_F3D = 70,
	SYSTEMZ_REG_F4D = 71,
	SYSTEMZ_REG_F5D = 72,
	SYSTEMZ_REG_F6D = 73,
	SYSTEMZ_REG_F7D = 74,
	SYSTEMZ_REG_F8D = 75,
	SYSTEMZ_REG_F9D = 76,
	SYSTEMZ_REG_F10D = 77,
	SYSTEMZ_REG_F11D = 78,
	SYSTEMZ_REG_F12D = 79,
	SYSTEMZ_REG_F13D = 80,
	SYSTEMZ_REG_F14D = 81,
	SYSTEMZ_REG_F15D = 82,
	SYSTEMZ_REG_F16D = 83,
	SYSTEMZ_REG_F17D = 84,
	SYSTEMZ_REG_F18D = 85,
	SYSTEMZ_REG_F19D = 86,
	SYSTEMZ_REG_F20D = 87,
	SYSTEMZ_REG_F21D = 88,
	SYSTEMZ_REG_F22D = 89,
	SYSTEMZ_REG_F23D = 90,
	SYSTEMZ_REG_F24D = 91,
	SYSTEMZ_REG_F25D = 92,
	SYSTEMZ_REG_F26D = 93,
	SYSTEMZ_REG_F27D = 94,
	SYSTEMZ_REG_F28D = 95,
	SYSTEMZ_REG_F29D = 96,
	SYSTEMZ_REG_F30D = 97,
	SYSTEMZ_REG_F31D = 98,
	SYSTEMZ_REG_F0Q = 99,
	SYSTEMZ_REG_F1Q = 100,
	SYSTEMZ_REG_F4Q = 101,
	SYSTEMZ_REG_F5Q = 102,
	SYSTEMZ_REG_F8Q = 103,
	SYSTEMZ_REG_F9Q = 104,
	SYSTEMZ_REG_F12Q = 105,
	SYSTEMZ_REG_F13Q = 106,
	SYSTEMZ_REG_F0S = 107,
	SYSTEMZ_REG_F1S = 108,
	SYSTEMZ_REG_F2S = 109,
	SYSTEMZ_REG_F3S = 110,
	SYSTEMZ_REG_F4S = 111,
	SYSTEMZ_REG_F5S = 112,
	SYSTEMZ_REG_F6S = 113,
	SYSTEMZ_REG_F7S = 114,
	SYSTEMZ_REG_F8S = 115,
	SYSTEMZ_REG_F9S = 116,
	SYSTEMZ_REG_F10S = 117,
	SYSTEMZ_REG_F11S = 118,
	SYSTEMZ_REG_F12S = 119,
	SYSTEMZ_REG_F13S = 120,
	SYSTEMZ_REG_F14S = 121,
	SYSTEMZ_REG_F15S = 122,
	SYSTEMZ_REG_F16S = 123,
	SYSTEMZ_REG_F17S = 124,
	SYSTEMZ_REG_F18S = 125,
	SYSTEMZ_REG_F19S = 126,
	SYSTEMZ_REG_F20S = 127,
	SYSTEMZ_REG_F21S = 128,
	SYSTEMZ_REG_F22S = 129,
	SYSTEMZ_REG_F23S = 130,
	SYSTEMZ_REG_F24S = 131,
	SYSTEMZ_REG_F25S = 132,
	SYSTEMZ_REG_F26S = 133,
	SYSTEMZ_REG_F27S = 134,
	SYSTEMZ_REG_F28S = 135,
	SYSTEMZ_REG_F29S = 136,
	SYSTEMZ_REG_F30S = 137,
	SYSTEMZ_REG_F31S = 138,
	SYSTEMZ_REG_R0D = 139,
	SYSTEMZ_REG_R1D = 140,
	SYSTEMZ_REG_R2D = 141,
	SYSTEMZ_REG_R3D = 142,
	SYSTEMZ_REG_R4D = 143,
	SYSTEMZ_REG_R5D = 144,
	SYSTEMZ_REG_R6D = 145,
	SYSTEMZ_REG_R7D = 146,
	SYSTEMZ_REG_R8D = 147,
	SYSTEMZ_REG_R9D = 148,
	SYSTEMZ_REG_R10D = 149,
	SYSTEMZ_REG_R11D = 150,
	SYSTEMZ_REG_R12D = 151,
	SYSTEMZ_REG_R13D = 152,
	SYSTEMZ_REG_R14D = 153,
	SYSTEMZ_REG_R15D = 154,
	SYSTEMZ_REG_R0H = 155,
	SYSTEMZ_REG_R1H = 156,
	SYSTEMZ_REG_R2H = 157,
	SYSTEMZ_REG_R3H = 158,
	SYSTEMZ_REG_R4H = 159,
	SYSTEMZ_REG_R5H = 160,
	SYSTEMZ_REG_R6H = 161,
	SYSTEMZ_REG_R7H = 162,
	SYSTEMZ_REG_R8H = 163,
	SYSTEMZ_REG_R9H = 164,
	SYSTEMZ_REG_R10H = 165,
	SYSTEMZ_REG_R11H = 166,
	SYSTEMZ_REG_R12H = 167,
	SYSTEMZ_REG_R13H = 168,
	SYSTEMZ_REG_R14H = 169,
	SYSTEMZ_REG_R15H = 170,
	SYSTEMZ_REG_R0L = 171,
	SYSTEMZ_REG_R1L = 172,
	SYSTEMZ_REG_R2L = 173,
	SYSTEMZ_REG_R3L = 174,
	SYSTEMZ_REG_R4L = 175,
	SYSTEMZ_REG_R5L = 176,
	SYSTEMZ_REG_R6L = 177,
	SYSTEMZ_REG_R7L = 178,
	SYSTEMZ_REG_R8L = 179,
	SYSTEMZ_REG_R9L = 180,
	SYSTEMZ_REG_R10L = 181,
	SYSTEMZ_REG_R11L = 182,
	SYSTEMZ_REG_R12L = 183,
	SYSTEMZ_REG_R13L = 184,
	SYSTEMZ_REG_R14L = 185,
	SYSTEMZ_REG_R15L = 186,
	SYSTEMZ_REG_R0Q = 187,
	SYSTEMZ_REG_R2Q = 188,
	SYSTEMZ_REG_R4Q = 189,
	SYSTEMZ_REG_R6Q = 190,
	SYSTEMZ_REG_R8Q = 191,
	SYSTEMZ_REG_R10Q = 192,
	SYSTEMZ_REG_R12Q = 193,
	SYSTEMZ_REG_R14Q = 194,
	SYSTEMZ_REG_ENDING, // 195

	// clang-format on
	// generated content <SystemZGenCSRegEnum.inc> end

	// alias registers
	// None
} systemz_reg;

typedef enum {
	SYSTEMZ_INSN_FORM_INVALID = 0,
	// generated content <SystemZGenCSInsnFormatsEnum.inc> begin
	// clang-format off

	SYSTEMZ_INSN_FORM_INSTRXA,
	SYSTEMZ_INSN_FORM_INSTRXE,
	SYSTEMZ_INSN_FORM_INSTRRE,
	SYSTEMZ_INSN_FORM_INSTRR,
	SYSTEMZ_INSN_FORM_INSTRRFA,
	SYSTEMZ_INSN_FORM_INSTRILA,
	SYSTEMZ_INSN_FORM_INSTRXYA,
	SYSTEMZ_INSN_FORM_INSTRIA,
	SYSTEMZ_INSN_FORM_INSTRIED,
	SYSTEMZ_INSN_FORM_INSTSIY,
	SYSTEMZ_INSN_FORM_INSTSSB,
	SYSTEMZ_INSN_FORM_INSTRXB,
	SYSTEMZ_INSN_FORM_INSTRXYB,
	SYSTEMZ_INSN_FORM_INSTSMI,
	SYSTEMZ_INSN_FORM_INSTMII,
	SYSTEMZ_INSN_FORM_INSTRIB,
	SYSTEMZ_INSN_FORM_INSTRILB,
	SYSTEMZ_INSN_FORM_INSTRIC,
	SYSTEMZ_INSN_FORM_INSTRILC,
	SYSTEMZ_INSN_FORM_INSTRSI,
	SYSTEMZ_INSN_FORM_INSTRIEE,
	SYSTEMZ_INSN_FORM_INSTRSA,
	SYSTEMZ_INSN_FORM_INSTRSYA,
	SYSTEMZ_INSN_FORM_INSTRRFE,
	SYSTEMZ_INSN_FORM_INSTRSLB,
	SYSTEMZ_INSN_FORM_INSTS,
	SYSTEMZ_INSN_FORM_INSTSIL,
	SYSTEMZ_INSN_FORM_INSTRIS,
	SYSTEMZ_INSN_FORM_INSTRIEC,
	SYSTEMZ_INSN_FORM_INSTRIEA,
	SYSTEMZ_INSN_FORM_INSTRRS,
	SYSTEMZ_INSN_FORM_INSTRIEB,
	SYSTEMZ_INSN_FORM_INSTRRFC,
	SYSTEMZ_INSN_FORM_INSTSSA,
	SYSTEMZ_INSN_FORM_INSTRSYB,
	SYSTEMZ_INSN_FORM_INSTSI,
	SYSTEMZ_INSN_FORM_INSTRSB,
	SYSTEMZ_INSN_FORM_INSTRRFB,
	SYSTEMZ_INSN_FORM_INSTRRFD,
	SYSTEMZ_INSN_FORM_INSTSSF,
	SYSTEMZ_INSN_FORM_INSTSSE,
	SYSTEMZ_INSN_FORM_INSTRIEG,
	SYSTEMZ_INSN_FORM_INSTRXF,
	SYSTEMZ_INSN_FORM_INSTRRD,
	SYSTEMZ_INSN_FORM_INSTSSD,
	SYSTEMZ_INSN_FORM_INSTIE,
	SYSTEMZ_INSN_FORM_INSTE,
	SYSTEMZ_INSN_FORM_INSTRIEF,
	SYSTEMZ_INSN_FORM_INSTSSC,
	SYSTEMZ_INSN_FORM_INSTI,
	SYSTEMZ_INSN_FORM_INSTRSLA,
	SYSTEMZ_INSN_FORM_INSTVRRC,
	SYSTEMZ_INSN_FORM_INSTVRRD,
	SYSTEMZ_INSN_FORM_INSTVRIF,
	SYSTEMZ_INSN_FORM_INSTVRRA,
	SYSTEMZ_INSN_FORM_INSTVRRB,
	SYSTEMZ_INSN_FORM_INSTVRRK,
	SYSTEMZ_INSN_FORM_INSTVRRH,
	SYSTEMZ_INSN_FORM_INSTVRRJ,
	SYSTEMZ_INSN_FORM_INSTVRRI,
	SYSTEMZ_INSN_FORM_INSTVRII,
	SYSTEMZ_INSN_FORM_INSTVRID,
	SYSTEMZ_INSN_FORM_INSTVRSA,
	SYSTEMZ_INSN_FORM_INSTVRRE,
	SYSTEMZ_INSN_FORM_INSTVRIE,
	SYSTEMZ_INSN_FORM_INSTVRIA,
	SYSTEMZ_INSN_FORM_INSTVRV,
	SYSTEMZ_INSN_FORM_INSTVRIB,
	SYSTEMZ_INSN_FORM_INSTVRX,
	SYSTEMZ_INSN_FORM_INSTVRSC,
	SYSTEMZ_INSN_FORM_INSTVRIH,
	SYSTEMZ_INSN_FORM_INSTVRSB,
	SYSTEMZ_INSN_FORM_INSTVSI,
	SYSTEMZ_INSN_FORM_INSTVRSD,
	SYSTEMZ_INSN_FORM_INSTVRRF,
	SYSTEMZ_INSN_FORM_INSTVRIG,
	SYSTEMZ_INSN_FORM_INSTVRIC,
	SYSTEMZ_INSN_FORM_INSTVRRG,

	// clang-format on
	// generated content <SystemZGenCSInsnFormatsEnum.inc> end

} systemz_insn_form;

typedef enum {
	SYSTEMZ_AM_INVALID = 0,
	SYSTEMZ_AM_BD, ///< Base and displacement are set.
	SYSTEMZ_AM_BDX, ///< Base, displacement and index register are set.
	SYSTEMZ_AM_BDL, ///< Base, displacement and length (immediate) are set.
	SYSTEMZ_AM_BDR, ///< Base, displacement and length (register) are set.
	SYSTEMZ_AM_BDV, ///< Base, displacement and index vector register are set.
} systemz_addr_mode;

typedef struct {
	systemz_insn_form form;
} systemz_suppl_info;

/// Instruction's operand referring to memory
/// This is associated with SYSTEMZ_OP_MEM operand type above
typedef struct systemz_op_mem {
	/// Address mode. Indicates which field below are set.
	systemz_addr_mode am;
	/// base register, can be safely interpreted as
	/// a value of type `systemz_reg`, but it is only
	/// one byte wide
	uint8_t /* systemz_reg */ base;
	uint8_t /* systemz_reg */
		index; ///< Index register, same conditions apply here
	uint64_t length; ///< Length component. Can be a register or immediate.
	int64_t disp; ///< displacement/offset value
} systemz_op_mem;

/// Instruction operand
typedef struct cs_systemz_op {
	systemz_op_type type; ///< operand type
	union {
		systemz_reg reg; ///< register value for REG operand
		int64_t imm; ///< immediate value for IMM operand
		systemz_op_mem mem; ///< base/disp value for MEM operand
	};
	cs_ac_type access; ///< R/W access of the operand.
	uint8_t imm_width; ///< Bit width of the immediate. 0 if not specified.
} cs_systemz_op;

#define NUM_SYSTEMZ_OPS 6

// Instruction structure
typedef struct cs_systemz {
	systemz_cc cc; ///< Code condition
	systemz_insn_form format; ///< The instruction format.
	/// Number of operands of this instruction,
	/// or 0 when instruction has no operand.
	uint8_t op_count;
	cs_systemz_op
		operands[NUM_SYSTEMZ_OPS]; ///< operands for this instruction.
} cs_systemz;

/// SystemZ instruction
typedef enum systemz_insn {
	// generated content <SystemZGenCSInsnEnum.inc> begin
	// clang-format off

	SYSTEMZ_INS_INVALID,
	SYSTEMZ_INS_A,
	SYSTEMZ_INS_AD,
	SYSTEMZ_INS_ADB,
	SYSTEMZ_INS_ADBR,
	SYSTEMZ_INS_ADR,
	SYSTEMZ_INS_ADTR,
	SYSTEMZ_INS_ADTRA,
	SYSTEMZ_INS_AE,
	SYSTEMZ_INS_AEB,
	SYSTEMZ_INS_AEBR,
	SYSTEMZ_INS_AER,
	SYSTEMZ_INS_AFI,
	SYSTEMZ_INS_AG,
	SYSTEMZ_INS_AGF,
	SYSTEMZ_INS_AGFI,
	SYSTEMZ_INS_AGFR,
	SYSTEMZ_INS_AGH,
	SYSTEMZ_INS_AGHI,
	SYSTEMZ_INS_AGHIK,
	SYSTEMZ_INS_AGR,
	SYSTEMZ_INS_AGRK,
	SYSTEMZ_INS_AGSI,
	SYSTEMZ_INS_AH,
	SYSTEMZ_INS_AHHHR,
	SYSTEMZ_INS_AHHLR,
	SYSTEMZ_INS_AHI,
	SYSTEMZ_INS_AHIK,
	SYSTEMZ_INS_AHY,
	SYSTEMZ_INS_AIH,
	SYSTEMZ_INS_AL,
	SYSTEMZ_INS_ALC,
	SYSTEMZ_INS_ALCG,
	SYSTEMZ_INS_ALCGR,
	SYSTEMZ_INS_ALCR,
	SYSTEMZ_INS_ALFI,
	SYSTEMZ_INS_ALG,
	SYSTEMZ_INS_ALGF,
	SYSTEMZ_INS_ALGFI,
	SYSTEMZ_INS_ALGFR,
	SYSTEMZ_INS_ALGHSIK,
	SYSTEMZ_INS_ALGR,
	SYSTEMZ_INS_ALGRK,
	SYSTEMZ_INS_ALGSI,
	SYSTEMZ_INS_ALHHHR,
	SYSTEMZ_INS_ALHHLR,
	SYSTEMZ_INS_ALHSIK,
	SYSTEMZ_INS_ALR,
	SYSTEMZ_INS_ALRK,
	SYSTEMZ_INS_ALSI,
	SYSTEMZ_INS_ALSIH,
	SYSTEMZ_INS_ALSIHN,
	SYSTEMZ_INS_ALY,
	SYSTEMZ_INS_AP,
	SYSTEMZ_INS_AR,
	SYSTEMZ_INS_ARK,
	SYSTEMZ_INS_ASI,
	SYSTEMZ_INS_AU,
	SYSTEMZ_INS_AUR,
	SYSTEMZ_INS_AW,
	SYSTEMZ_INS_AWR,
	SYSTEMZ_INS_AXBR,
	SYSTEMZ_INS_AXR,
	SYSTEMZ_INS_AXTR,
	SYSTEMZ_INS_AXTRA,
	SYSTEMZ_INS_AY,
	SYSTEMZ_INS_B,
	SYSTEMZ_INS_BAKR,
	SYSTEMZ_INS_BAL,
	SYSTEMZ_INS_BALR,
	SYSTEMZ_INS_BAS,
	SYSTEMZ_INS_BASR,
	SYSTEMZ_INS_BASSM,
	SYSTEMZ_INS_BE,
	SYSTEMZ_INS_BH,
	SYSTEMZ_INS_BHE,
	SYSTEMZ_INS_BL,
	SYSTEMZ_INS_BLE,
	SYSTEMZ_INS_BLH,
	SYSTEMZ_INS_BM,
	SYSTEMZ_INS_BNE,
	SYSTEMZ_INS_BNH,
	SYSTEMZ_INS_BNHE,
	SYSTEMZ_INS_BNL,
	SYSTEMZ_INS_BNLE,
	SYSTEMZ_INS_BNLH,
	SYSTEMZ_INS_BNM,
	SYSTEMZ_INS_BNO,
	SYSTEMZ_INS_BNP,
	SYSTEMZ_INS_BNZ,
	SYSTEMZ_INS_BO,
	SYSTEMZ_INS_BP,
	SYSTEMZ_INS_BZ,
	SYSTEMZ_INS_BC,
	SYSTEMZ_INS_BCR,
	SYSTEMZ_INS_BCT,
	SYSTEMZ_INS_BCTG,
	SYSTEMZ_INS_BCTGR,
	SYSTEMZ_INS_BCTR,
	SYSTEMZ_INS_BI,
	SYSTEMZ_INS_BIE,
	SYSTEMZ_INS_BIH,
	SYSTEMZ_INS_BIHE,
	SYSTEMZ_INS_BIL,
	SYSTEMZ_INS_BILE,
	SYSTEMZ_INS_BILH,
	SYSTEMZ_INS_BIM,
	SYSTEMZ_INS_BINE,
	SYSTEMZ_INS_BINH,
	SYSTEMZ_INS_BINHE,
	SYSTEMZ_INS_BINL,
	SYSTEMZ_INS_BINLE,
	SYSTEMZ_INS_BINLH,
	SYSTEMZ_INS_BINM,
	SYSTEMZ_INS_BINO,
	SYSTEMZ_INS_BINP,
	SYSTEMZ_INS_BINZ,
	SYSTEMZ_INS_BIO,
	SYSTEMZ_INS_BIP,
	SYSTEMZ_INS_BIZ,
	SYSTEMZ_INS_BIC,
	SYSTEMZ_INS_BPP,
	SYSTEMZ_INS_BPRP,
	SYSTEMZ_INS_BR,
	SYSTEMZ_INS_BRAS,
	SYSTEMZ_INS_BRASL,
	SYSTEMZ_INS_BER,
	SYSTEMZ_INS_BHR,
	SYSTEMZ_INS_BHER,
	SYSTEMZ_INS_BLR,
	SYSTEMZ_INS_BLER,
	SYSTEMZ_INS_BLHR,
	SYSTEMZ_INS_BMR,
	SYSTEMZ_INS_BNER,
	SYSTEMZ_INS_BNHR,
	SYSTEMZ_INS_BNHER,
	SYSTEMZ_INS_BNLR,
	SYSTEMZ_INS_BNLER,
	SYSTEMZ_INS_BNLHR,
	SYSTEMZ_INS_BNMR,
	SYSTEMZ_INS_BNOR,
	SYSTEMZ_INS_BNPR,
	SYSTEMZ_INS_BNZR,
	SYSTEMZ_INS_BOR,
	SYSTEMZ_INS_BPR,
	SYSTEMZ_INS_BZR,
	SYSTEMZ_INS_BRC,
	SYSTEMZ_INS_BRCL,
	SYSTEMZ_INS_BRCT,
	SYSTEMZ_INS_BRCTG,
	SYSTEMZ_INS_BRCTH,
	SYSTEMZ_INS_BRXH,
	SYSTEMZ_INS_BRXHG,
	SYSTEMZ_INS_BRXLE,
	SYSTEMZ_INS_BRXLG,
	SYSTEMZ_INS_BSA,
	SYSTEMZ_INS_BSG,
	SYSTEMZ_INS_BSM,
	SYSTEMZ_INS_BXH,
	SYSTEMZ_INS_BXHG,
	SYSTEMZ_INS_BXLE,
	SYSTEMZ_INS_BXLEG,
	SYSTEMZ_INS_C,
	SYSTEMZ_INS_CD,
	SYSTEMZ_INS_CDB,
	SYSTEMZ_INS_CDBR,
	SYSTEMZ_INS_CDFBR,
	SYSTEMZ_INS_CDFBRA,
	SYSTEMZ_INS_CDFR,
	SYSTEMZ_INS_CDFTR,
	SYSTEMZ_INS_CDGBR,
	SYSTEMZ_INS_CDGBRA,
	SYSTEMZ_INS_CDGR,
	SYSTEMZ_INS_CDGTR,
	SYSTEMZ_INS_CDGTRA,
	SYSTEMZ_INS_CDLFBR,
	SYSTEMZ_INS_CDLFTR,
	SYSTEMZ_INS_CDLGBR,
	SYSTEMZ_INS_CDLGTR,
	SYSTEMZ_INS_CDPT,
	SYSTEMZ_INS_CDR,
	SYSTEMZ_INS_CDS,
	SYSTEMZ_INS_CDSG,
	SYSTEMZ_INS_CDSTR,
	SYSTEMZ_INS_CDSY,
	SYSTEMZ_INS_CDTR,
	SYSTEMZ_INS_CDUTR,
	SYSTEMZ_INS_CDZT,
	SYSTEMZ_INS_CE,
	SYSTEMZ_INS_CEB,
	SYSTEMZ_INS_CEBR,
	SYSTEMZ_INS_CEDTR,
	SYSTEMZ_INS_CEFBR,
	SYSTEMZ_INS_CEFBRA,
	SYSTEMZ_INS_CEFR,
	SYSTEMZ_INS_CEGBR,
	SYSTEMZ_INS_CEGBRA,
	SYSTEMZ_INS_CEGR,
	SYSTEMZ_INS_CELFBR,
	SYSTEMZ_INS_CELGBR,
	SYSTEMZ_INS_CER,
	SYSTEMZ_INS_CEXTR,
	SYSTEMZ_INS_CFC,
	SYSTEMZ_INS_CFDBR,
	SYSTEMZ_INS_CFDBRA,
	SYSTEMZ_INS_CFDR,
	SYSTEMZ_INS_CFDTR,
	SYSTEMZ_INS_CFEBR,
	SYSTEMZ_INS_CFEBRA,
	SYSTEMZ_INS_CFER,
	SYSTEMZ_INS_CFI,
	SYSTEMZ_INS_CFXBR,
	SYSTEMZ_INS_CFXBRA,
	SYSTEMZ_INS_CFXR,
	SYSTEMZ_INS_CFXTR,
	SYSTEMZ_INS_CG,
	SYSTEMZ_INS_CGDBR,
	SYSTEMZ_INS_CGDBRA,
	SYSTEMZ_INS_CGDR,
	SYSTEMZ_INS_CGDTR,
	SYSTEMZ_INS_CGDTRA,
	SYSTEMZ_INS_CGEBR,
	SYSTEMZ_INS_CGEBRA,
	SYSTEMZ_INS_CGER,
	SYSTEMZ_INS_CGF,
	SYSTEMZ_INS_CGFI,
	SYSTEMZ_INS_CGFR,
	SYSTEMZ_INS_CGFRL,
	SYSTEMZ_INS_CGH,
	SYSTEMZ_INS_CGHI,
	SYSTEMZ_INS_CGHRL,
	SYSTEMZ_INS_CGHSI,
	SYSTEMZ_INS_CGIB,
	SYSTEMZ_INS_CGIBE,
	SYSTEMZ_INS_CGIBH,
	SYSTEMZ_INS_CGIBHE,
	SYSTEMZ_INS_CGIBL,
	SYSTEMZ_INS_CGIBLE,
	SYSTEMZ_INS_CGIBLH,
	SYSTEMZ_INS_CGIBNE,
	SYSTEMZ_INS_CGIBNH,
	SYSTEMZ_INS_CGIBNHE,
	SYSTEMZ_INS_CGIBNL,
	SYSTEMZ_INS_CGIBNLE,
	SYSTEMZ_INS_CGIBNLH,
	SYSTEMZ_INS_CGIJ,
	SYSTEMZ_INS_CGIJE,
	SYSTEMZ_INS_CGIJH,
	SYSTEMZ_INS_CGIJHE,
	SYSTEMZ_INS_CGIJL,
	SYSTEMZ_INS_CGIJLE,
	SYSTEMZ_INS_CGIJLH,
	SYSTEMZ_INS_CGIJNE,
	SYSTEMZ_INS_CGIJNH,
	SYSTEMZ_INS_CGIJNHE,
	SYSTEMZ_INS_CGIJNL,
	SYSTEMZ_INS_CGIJNLE,
	SYSTEMZ_INS_CGIJNLH,
	SYSTEMZ_INS_CGIT,
	SYSTEMZ_INS_CGITE,
	SYSTEMZ_INS_CGITH,
	SYSTEMZ_INS_CGITHE,
	SYSTEMZ_INS_CGITL,
	SYSTEMZ_INS_CGITLE,
	SYSTEMZ_INS_CGITLH,
	SYSTEMZ_INS_CGITNE,
	SYSTEMZ_INS_CGITNH,
	SYSTEMZ_INS_CGITNHE,
	SYSTEMZ_INS_CGITNL,
	SYSTEMZ_INS_CGITNLE,
	SYSTEMZ_INS_CGITNLH,
	SYSTEMZ_INS_CGR,
	SYSTEMZ_INS_CGRB,
	SYSTEMZ_INS_CGRBE,
	SYSTEMZ_INS_CGRBH,
	SYSTEMZ_INS_CGRBHE,
	SYSTEMZ_INS_CGRBL,
	SYSTEMZ_INS_CGRBLE,
	SYSTEMZ_INS_CGRBLH,
	SYSTEMZ_INS_CGRBNE,
	SYSTEMZ_INS_CGRBNH,
	SYSTEMZ_INS_CGRBNHE,
	SYSTEMZ_INS_CGRBNL,
	SYSTEMZ_INS_CGRBNLE,
	SYSTEMZ_INS_CGRBNLH,
	SYSTEMZ_INS_CGRJ,
	SYSTEMZ_INS_CGRJE,
	SYSTEMZ_INS_CGRJH,
	SYSTEMZ_INS_CGRJHE,
	SYSTEMZ_INS_CGRJL,
	SYSTEMZ_INS_CGRJLE,
	SYSTEMZ_INS_CGRJLH,
	SYSTEMZ_INS_CGRJNE,
	SYSTEMZ_INS_CGRJNH,
	SYSTEMZ_INS_CGRJNHE,
	SYSTEMZ_INS_CGRJNL,
	SYSTEMZ_INS_CGRJNLE,
	SYSTEMZ_INS_CGRJNLH,
	SYSTEMZ_INS_CGRL,
	SYSTEMZ_INS_CGRT,
	SYSTEMZ_INS_CGRTE,
	SYSTEMZ_INS_CGRTH,
	SYSTEMZ_INS_CGRTHE,
	SYSTEMZ_INS_CGRTL,
	SYSTEMZ_INS_CGRTLE,
	SYSTEMZ_INS_CGRTLH,
	SYSTEMZ_INS_CGRTNE,
	SYSTEMZ_INS_CGRTNH,
	SYSTEMZ_INS_CGRTNHE,
	SYSTEMZ_INS_CGRTNL,
	SYSTEMZ_INS_CGRTNLE,
	SYSTEMZ_INS_CGRTNLH,
	SYSTEMZ_INS_CGXBR,
	SYSTEMZ_INS_CGXBRA,
	SYSTEMZ_INS_CGXR,
	SYSTEMZ_INS_CGXTR,
	SYSTEMZ_INS_CGXTRA,
	SYSTEMZ_INS_CH,
	SYSTEMZ_INS_CHF,
	SYSTEMZ_INS_CHHR,
	SYSTEMZ_INS_CHHSI,
	SYSTEMZ_INS_CHI,
	SYSTEMZ_INS_CHLR,
	SYSTEMZ_INS_CHRL,
	SYSTEMZ_INS_CHSI,
	SYSTEMZ_INS_CHY,
	SYSTEMZ_INS_CIB,
	SYSTEMZ_INS_CIBE,
	SYSTEMZ_INS_CIBH,
	SYSTEMZ_INS_CIBHE,
	SYSTEMZ_INS_CIBL,
	SYSTEMZ_INS_CIBLE,
	SYSTEMZ_INS_CIBLH,
	SYSTEMZ_INS_CIBNE,
	SYSTEMZ_INS_CIBNH,
	SYSTEMZ_INS_CIBNHE,
	SYSTEMZ_INS_CIBNL,
	SYSTEMZ_INS_CIBNLE,
	SYSTEMZ_INS_CIBNLH,
	SYSTEMZ_INS_CIH,
	SYSTEMZ_INS_CIJ,
	SYSTEMZ_INS_CIJE,
	SYSTEMZ_INS_CIJH,
	SYSTEMZ_INS_CIJHE,
	SYSTEMZ_INS_CIJL,
	SYSTEMZ_INS_CIJLE,
	SYSTEMZ_INS_CIJLH,
	SYSTEMZ_INS_CIJNE,
	SYSTEMZ_INS_CIJNH,
	SYSTEMZ_INS_CIJNHE,
	SYSTEMZ_INS_CIJNL,
	SYSTEMZ_INS_CIJNLE,
	SYSTEMZ_INS_CIJNLH,
	SYSTEMZ_INS_CIT,
	SYSTEMZ_INS_CITE,
	SYSTEMZ_INS_CITH,
	SYSTEMZ_INS_CITHE,
	SYSTEMZ_INS_CITL,
	SYSTEMZ_INS_CITLE,
	SYSTEMZ_INS_CITLH,
	SYSTEMZ_INS_CITNE,
	SYSTEMZ_INS_CITNH,
	SYSTEMZ_INS_CITNHE,
	SYSTEMZ_INS_CITNL,
	SYSTEMZ_INS_CITNLE,
	SYSTEMZ_INS_CITNLH,
	SYSTEMZ_INS_CKSM,
	SYSTEMZ_INS_CL,
	SYSTEMZ_INS_CLC,
	SYSTEMZ_INS_CLCL,
	SYSTEMZ_INS_CLCLE,
	SYSTEMZ_INS_CLCLU,
	SYSTEMZ_INS_CLFDBR,
	SYSTEMZ_INS_CLFDTR,
	SYSTEMZ_INS_CLFEBR,
	SYSTEMZ_INS_CLFHSI,
	SYSTEMZ_INS_CLFI,
	SYSTEMZ_INS_CLFIT,
	SYSTEMZ_INS_CLFITE,
	SYSTEMZ_INS_CLFITH,
	SYSTEMZ_INS_CLFITHE,
	SYSTEMZ_INS_CLFITL,
	SYSTEMZ_INS_CLFITLE,
	SYSTEMZ_INS_CLFITLH,
	SYSTEMZ_INS_CLFITNE,
	SYSTEMZ_INS_CLFITNH,
	SYSTEMZ_INS_CLFITNHE,
	SYSTEMZ_INS_CLFITNL,
	SYSTEMZ_INS_CLFITNLE,
	SYSTEMZ_INS_CLFITNLH,
	SYSTEMZ_INS_CLFXBR,
	SYSTEMZ_INS_CLFXTR,
	SYSTEMZ_INS_CLG,
	SYSTEMZ_INS_CLGDBR,
	SYSTEMZ_INS_CLGDTR,
	SYSTEMZ_INS_CLGEBR,
	SYSTEMZ_INS_CLGF,
	SYSTEMZ_INS_CLGFI,
	SYSTEMZ_INS_CLGFR,
	SYSTEMZ_INS_CLGFRL,
	SYSTEMZ_INS_CLGHRL,
	SYSTEMZ_INS_CLGHSI,
	SYSTEMZ_INS_CLGIB,
	SYSTEMZ_INS_CLGIBE,
	SYSTEMZ_INS_CLGIBH,
	SYSTEMZ_INS_CLGIBHE,
	SYSTEMZ_INS_CLGIBL,
	SYSTEMZ_INS_CLGIBLE,
	SYSTEMZ_INS_CLGIBLH,
	SYSTEMZ_INS_CLGIBNE,
	SYSTEMZ_INS_CLGIBNH,
	SYSTEMZ_INS_CLGIBNHE,
	SYSTEMZ_INS_CLGIBNL,
	SYSTEMZ_INS_CLGIBNLE,
	SYSTEMZ_INS_CLGIBNLH,
	SYSTEMZ_INS_CLGIJ,
	SYSTEMZ_INS_CLGIJE,
	SYSTEMZ_INS_CLGIJH,
	SYSTEMZ_INS_CLGIJHE,
	SYSTEMZ_INS_CLGIJL,
	SYSTEMZ_INS_CLGIJLE,
	SYSTEMZ_INS_CLGIJLH,
	SYSTEMZ_INS_CLGIJNE,
	SYSTEMZ_INS_CLGIJNH,
	SYSTEMZ_INS_CLGIJNHE,
	SYSTEMZ_INS_CLGIJNL,
	SYSTEMZ_INS_CLGIJNLE,
	SYSTEMZ_INS_CLGIJNLH,
	SYSTEMZ_INS_CLGIT,
	SYSTEMZ_INS_CLGITE,
	SYSTEMZ_INS_CLGITH,
	SYSTEMZ_INS_CLGITHE,
	SYSTEMZ_INS_CLGITL,
	SYSTEMZ_INS_CLGITLE,
	SYSTEMZ_INS_CLGITLH,
	SYSTEMZ_INS_CLGITNE,
	SYSTEMZ_INS_CLGITNH,
	SYSTEMZ_INS_CLGITNHE,
	SYSTEMZ_INS_CLGITNL,
	SYSTEMZ_INS_CLGITNLE,
	SYSTEMZ_INS_CLGITNLH,
	SYSTEMZ_INS_CLGR,
	SYSTEMZ_INS_CLGRB,
	SYSTEMZ_INS_CLGRBE,
	SYSTEMZ_INS_CLGRBH,
	SYSTEMZ_INS_CLGRBHE,
	SYSTEMZ_INS_CLGRBL,
	SYSTEMZ_INS_CLGRBLE,
	SYSTEMZ_INS_CLGRBLH,
	SYSTEMZ_INS_CLGRBNE,
	SYSTEMZ_INS_CLGRBNH,
	SYSTEMZ_INS_CLGRBNHE,
	SYSTEMZ_INS_CLGRBNL,
	SYSTEMZ_INS_CLGRBNLE,
	SYSTEMZ_INS_CLGRBNLH,
	SYSTEMZ_INS_CLGRJ,
	SYSTEMZ_INS_CLGRJE,
	SYSTEMZ_INS_CLGRJH,
	SYSTEMZ_INS_CLGRJHE,
	SYSTEMZ_INS_CLGRJL,
	SYSTEMZ_INS_CLGRJLE,
	SYSTEMZ_INS_CLGRJLH,
	SYSTEMZ_INS_CLGRJNE,
	SYSTEMZ_INS_CLGRJNH,
	SYSTEMZ_INS_CLGRJNHE,
	SYSTEMZ_INS_CLGRJNL,
	SYSTEMZ_INS_CLGRJNLE,
	SYSTEMZ_INS_CLGRJNLH,
	SYSTEMZ_INS_CLGRL,
	SYSTEMZ_INS_CLGRT,
	SYSTEMZ_INS_CLGRTE,
	SYSTEMZ_INS_CLGRTH,
	SYSTEMZ_INS_CLGRTHE,
	SYSTEMZ_INS_CLGRTL,
	SYSTEMZ_INS_CLGRTLE,
	SYSTEMZ_INS_CLGRTLH,
	SYSTEMZ_INS_CLGRTNE,
	SYSTEMZ_INS_CLGRTNH,
	SYSTEMZ_INS_CLGRTNHE,
	SYSTEMZ_INS_CLGRTNL,
	SYSTEMZ_INS_CLGRTNLE,
	SYSTEMZ_INS_CLGRTNLH,
	SYSTEMZ_INS_CLGT,
	SYSTEMZ_INS_CLGTE,
	SYSTEMZ_INS_CLGTH,
	SYSTEMZ_INS_CLGTHE,
	SYSTEMZ_INS_CLGTL,
	SYSTEMZ_INS_CLGTLE,
	SYSTEMZ_INS_CLGTLH,
	SYSTEMZ_INS_CLGTNE,
	SYSTEMZ_INS_CLGTNH,
	SYSTEMZ_INS_CLGTNHE,
	SYSTEMZ_INS_CLGTNL,
	SYSTEMZ_INS_CLGTNLE,
	SYSTEMZ_INS_CLGTNLH,
	SYSTEMZ_INS_CLGXBR,
	SYSTEMZ_INS_CLGXTR,
	SYSTEMZ_INS_CLHF,
	SYSTEMZ_INS_CLHHR,
	SYSTEMZ_INS_CLHHSI,
	SYSTEMZ_INS_CLHLR,
	SYSTEMZ_INS_CLHRL,
	SYSTEMZ_INS_CLI,
	SYSTEMZ_INS_CLIB,
	SYSTEMZ_INS_CLIBE,
	SYSTEMZ_INS_CLIBH,
	SYSTEMZ_INS_CLIBHE,
	SYSTEMZ_INS_CLIBL,
	SYSTEMZ_INS_CLIBLE,
	SYSTEMZ_INS_CLIBLH,
	SYSTEMZ_INS_CLIBNE,
	SYSTEMZ_INS_CLIBNH,
	SYSTEMZ_INS_CLIBNHE,
	SYSTEMZ_INS_CLIBNL,
	SYSTEMZ_INS_CLIBNLE,
	SYSTEMZ_INS_CLIBNLH,
	SYSTEMZ_INS_CLIH,
	SYSTEMZ_INS_CLIJ,
	SYSTEMZ_INS_CLIJE,
	SYSTEMZ_INS_CLIJH,
	SYSTEMZ_INS_CLIJHE,
	SYSTEMZ_INS_CLIJL,
	SYSTEMZ_INS_CLIJLE,
	SYSTEMZ_INS_CLIJLH,
	SYSTEMZ_INS_CLIJNE,
	SYSTEMZ_INS_CLIJNH,
	SYSTEMZ_INS_CLIJNHE,
	SYSTEMZ_INS_CLIJNL,
	SYSTEMZ_INS_CLIJNLE,
	SYSTEMZ_INS_CLIJNLH,
	SYSTEMZ_INS_CLIY,
	SYSTEMZ_INS_CLM,
	SYSTEMZ_INS_CLMH,
	SYSTEMZ_INS_CLMY,
	SYSTEMZ_INS_CLR,
	SYSTEMZ_INS_CLRB,
	SYSTEMZ_INS_CLRBE,
	SYSTEMZ_INS_CLRBH,
	SYSTEMZ_INS_CLRBHE,
	SYSTEMZ_INS_CLRBL,
	SYSTEMZ_INS_CLRBLE,
	SYSTEMZ_INS_CLRBLH,
	SYSTEMZ_INS_CLRBNE,
	SYSTEMZ_INS_CLRBNH,
	SYSTEMZ_INS_CLRBNHE,
	SYSTEMZ_INS_CLRBNL,
	SYSTEMZ_INS_CLRBNLE,
	SYSTEMZ_INS_CLRBNLH,
	SYSTEMZ_INS_CLRJ,
	SYSTEMZ_INS_CLRJE,
	SYSTEMZ_INS_CLRJH,
	SYSTEMZ_INS_CLRJHE,
	SYSTEMZ_INS_CLRJL,
	SYSTEMZ_INS_CLRJLE,
	SYSTEMZ_INS_CLRJLH,
	SYSTEMZ_INS_CLRJNE,
	SYSTEMZ_INS_CLRJNH,
	SYSTEMZ_INS_CLRJNHE,
	SYSTEMZ_INS_CLRJNL,
	SYSTEMZ_INS_CLRJNLE,
	SYSTEMZ_INS_CLRJNLH,
	SYSTEMZ_INS_CLRL,
	SYSTEMZ_INS_CLRT,
	SYSTEMZ_INS_CLRTE,
	SYSTEMZ_INS_CLRTH,
	SYSTEMZ_INS_CLRTHE,
	SYSTEMZ_INS_CLRTL,
	SYSTEMZ_INS_CLRTLE,
	SYSTEMZ_INS_CLRTLH,
	SYSTEMZ_INS_CLRTNE,
	SYSTEMZ_INS_CLRTNH,
	SYSTEMZ_INS_CLRTNHE,
	SYSTEMZ_INS_CLRTNL,
	SYSTEMZ_INS_CLRTNLE,
	SYSTEMZ_INS_CLRTNLH,
	SYSTEMZ_INS_CLST,
	SYSTEMZ_INS_CLT,
	SYSTEMZ_INS_CLTE,
	SYSTEMZ_INS_CLTH,
	SYSTEMZ_INS_CLTHE,
	SYSTEMZ_INS_CLTL,
	SYSTEMZ_INS_CLTLE,
	SYSTEMZ_INS_CLTLH,
	SYSTEMZ_INS_CLTNE,
	SYSTEMZ_INS_CLTNH,
	SYSTEMZ_INS_CLTNHE,
	SYSTEMZ_INS_CLTNL,
	SYSTEMZ_INS_CLTNLE,
	SYSTEMZ_INS_CLTNLH,
	SYSTEMZ_INS_CLY,
	SYSTEMZ_INS_CMPSC,
	SYSTEMZ_INS_CP,
	SYSTEMZ_INS_CPDT,
	SYSTEMZ_INS_CPSDR,
	SYSTEMZ_INS_CPXT,
	SYSTEMZ_INS_CPYA,
	SYSTEMZ_INS_CR,
	SYSTEMZ_INS_CRB,
	SYSTEMZ_INS_CRBE,
	SYSTEMZ_INS_CRBH,
	SYSTEMZ_INS_CRBHE,
	SYSTEMZ_INS_CRBL,
	SYSTEMZ_INS_CRBLE,
	SYSTEMZ_INS_CRBLH,
	SYSTEMZ_INS_CRBNE,
	SYSTEMZ_INS_CRBNH,
	SYSTEMZ_INS_CRBNHE,
	SYSTEMZ_INS_CRBNL,
	SYSTEMZ_INS_CRBNLE,
	SYSTEMZ_INS_CRBNLH,
	SYSTEMZ_INS_CRDTE,
	SYSTEMZ_INS_CRJ,
	SYSTEMZ_INS_CRJE,
	SYSTEMZ_INS_CRJH,
	SYSTEMZ_INS_CRJHE,
	SYSTEMZ_INS_CRJL,
	SYSTEMZ_INS_CRJLE,
	SYSTEMZ_INS_CRJLH,
	SYSTEMZ_INS_CRJNE,
	SYSTEMZ_INS_CRJNH,
	SYSTEMZ_INS_CRJNHE,
	SYSTEMZ_INS_CRJNL,
	SYSTEMZ_INS_CRJNLE,
	SYSTEMZ_INS_CRJNLH,
	SYSTEMZ_INS_CRL,
	SYSTEMZ_INS_CRT,
	SYSTEMZ_INS_CRTE,
	SYSTEMZ_INS_CRTH,
	SYSTEMZ_INS_CRTHE,
	SYSTEMZ_INS_CRTL,
	SYSTEMZ_INS_CRTLE,
	SYSTEMZ_INS_CRTLH,
	SYSTEMZ_INS_CRTNE,
	SYSTEMZ_INS_CRTNH,
	SYSTEMZ_INS_CRTNHE,
	SYSTEMZ_INS_CRTNL,
	SYSTEMZ_INS_CRTNLE,
	SYSTEMZ_INS_CRTNLH,
	SYSTEMZ_INS_CS,
	SYSTEMZ_INS_CSCH,
	SYSTEMZ_INS_CSDTR,
	SYSTEMZ_INS_CSG,
	SYSTEMZ_INS_CSP,
	SYSTEMZ_INS_CSPG,
	SYSTEMZ_INS_CSST,
	SYSTEMZ_INS_CSXTR,
	SYSTEMZ_INS_CSY,
	SYSTEMZ_INS_CU12,
	SYSTEMZ_INS_CU14,
	SYSTEMZ_INS_CU21,
	SYSTEMZ_INS_CU24,
	SYSTEMZ_INS_CU41,
	SYSTEMZ_INS_CU42,
	SYSTEMZ_INS_CUDTR,
	SYSTEMZ_INS_CUSE,
	SYSTEMZ_INS_CUTFU,
	SYSTEMZ_INS_CUUTF,
	SYSTEMZ_INS_CUXTR,
	SYSTEMZ_INS_CVB,
	SYSTEMZ_INS_CVBG,
	SYSTEMZ_INS_CVBY,
	SYSTEMZ_INS_CVD,
	SYSTEMZ_INS_CVDG,
	SYSTEMZ_INS_CVDY,
	SYSTEMZ_INS_CXBR,
	SYSTEMZ_INS_CXFBR,
	SYSTEMZ_INS_CXFBRA,
	SYSTEMZ_INS_CXFR,
	SYSTEMZ_INS_CXFTR,
	SYSTEMZ_INS_CXGBR,
	SYSTEMZ_INS_CXGBRA,
	SYSTEMZ_INS_CXGR,
	SYSTEMZ_INS_CXGTR,
	SYSTEMZ_INS_CXGTRA,
	SYSTEMZ_INS_CXLFBR,
	SYSTEMZ_INS_CXLFTR,
	SYSTEMZ_INS_CXLGBR,
	SYSTEMZ_INS_CXLGTR,
	SYSTEMZ_INS_CXPT,
	SYSTEMZ_INS_CXR,
	SYSTEMZ_INS_CXSTR,
	SYSTEMZ_INS_CXTR,
	SYSTEMZ_INS_CXUTR,
	SYSTEMZ_INS_CXZT,
	SYSTEMZ_INS_CY,
	SYSTEMZ_INS_CZDT,
	SYSTEMZ_INS_CZXT,
	SYSTEMZ_INS_D,
	SYSTEMZ_INS_DD,
	SYSTEMZ_INS_DDB,
	SYSTEMZ_INS_DDBR,
	SYSTEMZ_INS_DDR,
	SYSTEMZ_INS_DDTR,
	SYSTEMZ_INS_DDTRA,
	SYSTEMZ_INS_DE,
	SYSTEMZ_INS_DEB,
	SYSTEMZ_INS_DEBR,
	SYSTEMZ_INS_DER,
	SYSTEMZ_INS_DFLTCC,
	SYSTEMZ_INS_DIAG,
	SYSTEMZ_INS_DIDBR,
	SYSTEMZ_INS_DIEBR,
	SYSTEMZ_INS_DL,
	SYSTEMZ_INS_DLG,
	SYSTEMZ_INS_DLGR,
	SYSTEMZ_INS_DLR,
	SYSTEMZ_INS_DP,
	SYSTEMZ_INS_DR,
	SYSTEMZ_INS_DSG,
	SYSTEMZ_INS_DSGF,
	SYSTEMZ_INS_DSGFR,
	SYSTEMZ_INS_DSGR,
	SYSTEMZ_INS_DXBR,
	SYSTEMZ_INS_DXR,
	SYSTEMZ_INS_DXTR,
	SYSTEMZ_INS_DXTRA,
	SYSTEMZ_INS_EAR,
	SYSTEMZ_INS_ECAG,
	SYSTEMZ_INS_ECCTR,
	SYSTEMZ_INS_ECPGA,
	SYSTEMZ_INS_ECTG,
	SYSTEMZ_INS_ED,
	SYSTEMZ_INS_EDMK,
	SYSTEMZ_INS_EEDTR,
	SYSTEMZ_INS_EEXTR,
	SYSTEMZ_INS_EFPC,
	SYSTEMZ_INS_EPAIR,
	SYSTEMZ_INS_EPAR,
	SYSTEMZ_INS_EPCTR,
	SYSTEMZ_INS_EPSW,
	SYSTEMZ_INS_EREG,
	SYSTEMZ_INS_EREGG,
	SYSTEMZ_INS_ESAIR,
	SYSTEMZ_INS_ESAR,
	SYSTEMZ_INS_ESDTR,
	SYSTEMZ_INS_ESEA,
	SYSTEMZ_INS_ESTA,
	SYSTEMZ_INS_ESXTR,
	SYSTEMZ_INS_ETND,
	SYSTEMZ_INS_EX,
	SYSTEMZ_INS_EXRL,
	SYSTEMZ_INS_FIDBR,
	SYSTEMZ_INS_FIDBRA,
	SYSTEMZ_INS_FIDR,
	SYSTEMZ_INS_FIDTR,
	SYSTEMZ_INS_FIEBR,
	SYSTEMZ_INS_FIEBRA,
	SYSTEMZ_INS_FIER,
	SYSTEMZ_INS_FIXBR,
	SYSTEMZ_INS_FIXBRA,
	SYSTEMZ_INS_FIXR,
	SYSTEMZ_INS_FIXTR,
	SYSTEMZ_INS_FLOGR,
	SYSTEMZ_INS_HDR,
	SYSTEMZ_INS_HER,
	SYSTEMZ_INS_HSCH,
	SYSTEMZ_INS_IAC,
	SYSTEMZ_INS_IC,
	SYSTEMZ_INS_ICM,
	SYSTEMZ_INS_ICMH,
	SYSTEMZ_INS_ICMY,
	SYSTEMZ_INS_ICY,
	SYSTEMZ_INS_IDTE,
	SYSTEMZ_INS_IEDTR,
	SYSTEMZ_INS_IEXTR,
	SYSTEMZ_INS_IIHF,
	SYSTEMZ_INS_IIHH,
	SYSTEMZ_INS_IIHL,
	SYSTEMZ_INS_IILF,
	SYSTEMZ_INS_IILH,
	SYSTEMZ_INS_IILL,
	SYSTEMZ_INS_IPK,
	SYSTEMZ_INS_IPM,
	SYSTEMZ_INS_IPTE,
	SYSTEMZ_INS_IRBM,
	SYSTEMZ_INS_ISKE,
	SYSTEMZ_INS_IVSK,
	SYSTEMZ_INS_J,
	SYSTEMZ_INS_JE,
	SYSTEMZ_INS_JH,
	SYSTEMZ_INS_JHE,
	SYSTEMZ_INS_JL,
	SYSTEMZ_INS_JLE,
	SYSTEMZ_INS_JLH,
	SYSTEMZ_INS_JM,
	SYSTEMZ_INS_JNE,
	SYSTEMZ_INS_JNH,
	SYSTEMZ_INS_JNHE,
	SYSTEMZ_INS_JNL,
	SYSTEMZ_INS_JNLE,
	SYSTEMZ_INS_JNLH,
	SYSTEMZ_INS_JNM,
	SYSTEMZ_INS_JNO,
	SYSTEMZ_INS_JNP,
	SYSTEMZ_INS_JNZ,
	SYSTEMZ_INS_JO,
	SYSTEMZ_INS_JP,
	SYSTEMZ_INS_JZ,
	SYSTEMZ_INS_J_G_LU_,
	SYSTEMZ_INS_J_G_L_E,
	SYSTEMZ_INS_J_G_L_H,
	SYSTEMZ_INS_J_G_L_HE,
	SYSTEMZ_INS_J_G_L_L,
	SYSTEMZ_INS_J_G_L_LE,
	SYSTEMZ_INS_J_G_L_LH,
	SYSTEMZ_INS_J_G_L_M,
	SYSTEMZ_INS_J_G_L_NE,
	SYSTEMZ_INS_J_G_L_NH,
	SYSTEMZ_INS_J_G_L_NHE,
	SYSTEMZ_INS_J_G_L_NL,
	SYSTEMZ_INS_J_G_L_NLE,
	SYSTEMZ_INS_J_G_L_NLH,
	SYSTEMZ_INS_J_G_L_NM,
	SYSTEMZ_INS_J_G_L_NO,
	SYSTEMZ_INS_J_G_L_NP,
	SYSTEMZ_INS_J_G_L_NZ,
	SYSTEMZ_INS_J_G_L_O,
	SYSTEMZ_INS_J_G_L_P,
	SYSTEMZ_INS_J_G_L_Z,
	SYSTEMZ_INS_KDB,
	SYSTEMZ_INS_KDBR,
	SYSTEMZ_INS_KDSA,
	SYSTEMZ_INS_KDTR,
	SYSTEMZ_INS_KEB,
	SYSTEMZ_INS_KEBR,
	SYSTEMZ_INS_KIMD,
	SYSTEMZ_INS_KLMD,
	SYSTEMZ_INS_KM,
	SYSTEMZ_INS_KMA,
	SYSTEMZ_INS_KMAC,
	SYSTEMZ_INS_KMC,
	SYSTEMZ_INS_KMCTR,
	SYSTEMZ_INS_KMF,
	SYSTEMZ_INS_KMO,
	SYSTEMZ_INS_KXBR,
	SYSTEMZ_INS_KXTR,
	SYSTEMZ_INS_L,
	SYSTEMZ_INS_LA,
	SYSTEMZ_INS_LAA,
	SYSTEMZ_INS_LAAG,
	SYSTEMZ_INS_LAAL,
	SYSTEMZ_INS_LAALG,
	SYSTEMZ_INS_LAE,
	SYSTEMZ_INS_LAEY,
	SYSTEMZ_INS_LAM,
	SYSTEMZ_INS_LAMY,
	SYSTEMZ_INS_LAN,
	SYSTEMZ_INS_LANG,
	SYSTEMZ_INS_LAO,
	SYSTEMZ_INS_LAOG,
	SYSTEMZ_INS_LARL,
	SYSTEMZ_INS_LASP,
	SYSTEMZ_INS_LAT,
	SYSTEMZ_INS_LAX,
	SYSTEMZ_INS_LAXG,
	SYSTEMZ_INS_LAY,
	SYSTEMZ_INS_LB,
	SYSTEMZ_INS_LBEAR,
	SYSTEMZ_INS_LBH,
	SYSTEMZ_INS_LBR,
	SYSTEMZ_INS_LCBB,
	SYSTEMZ_INS_LCCTL,
	SYSTEMZ_INS_LCDBR,
	SYSTEMZ_INS_LCDFR,
	SYSTEMZ_INS_LCDR,
	SYSTEMZ_INS_LCEBR,
	SYSTEMZ_INS_LCER,
	SYSTEMZ_INS_LCGFR,
	SYSTEMZ_INS_LCGR,
	SYSTEMZ_INS_LCR,
	SYSTEMZ_INS_LCTL,
	SYSTEMZ_INS_LCTLG,
	SYSTEMZ_INS_LCXBR,
	SYSTEMZ_INS_LCXR,
	SYSTEMZ_INS_LD,
	SYSTEMZ_INS_LDE,
	SYSTEMZ_INS_LDEB,
	SYSTEMZ_INS_LDEBR,
	SYSTEMZ_INS_LDER,
	SYSTEMZ_INS_LDETR,
	SYSTEMZ_INS_LDGR,
	SYSTEMZ_INS_LDR,
	SYSTEMZ_INS_LDXBR,
	SYSTEMZ_INS_LDXBRA,
	SYSTEMZ_INS_LDXR,
	SYSTEMZ_INS_LDXTR,
	SYSTEMZ_INS_LDY,
	SYSTEMZ_INS_LE,
	SYSTEMZ_INS_LEDBR,
	SYSTEMZ_INS_LEDBRA,
	SYSTEMZ_INS_LEDR,
	SYSTEMZ_INS_LEDTR,
	SYSTEMZ_INS_LER,
	SYSTEMZ_INS_LEXBR,
	SYSTEMZ_INS_LEXBRA,
	SYSTEMZ_INS_LEXR,
	SYSTEMZ_INS_LEY,
	SYSTEMZ_INS_LFAS,
	SYSTEMZ_INS_LFH,
	SYSTEMZ_INS_LFHAT,
	SYSTEMZ_INS_LFPC,
	SYSTEMZ_INS_LG,
	SYSTEMZ_INS_LGAT,
	SYSTEMZ_INS_LGB,
	SYSTEMZ_INS_LGBR,
	SYSTEMZ_INS_LGDR,
	SYSTEMZ_INS_LGF,
	SYSTEMZ_INS_LGFI,
	SYSTEMZ_INS_LGFR,
	SYSTEMZ_INS_LGFRL,
	SYSTEMZ_INS_LGG,
	SYSTEMZ_INS_LGH,
	SYSTEMZ_INS_LGHI,
	SYSTEMZ_INS_LGHR,
	SYSTEMZ_INS_LGHRL,
	SYSTEMZ_INS_LGR,
	SYSTEMZ_INS_LGRL,
	SYSTEMZ_INS_LGSC,
	SYSTEMZ_INS_LH,
	SYSTEMZ_INS_LHH,
	SYSTEMZ_INS_LHI,
	SYSTEMZ_INS_LHR,
	SYSTEMZ_INS_LHRL,
	SYSTEMZ_INS_LHY,
	SYSTEMZ_INS_LLC,
	SYSTEMZ_INS_LLCH,
	SYSTEMZ_INS_LLCR,
	SYSTEMZ_INS_LLGC,
	SYSTEMZ_INS_LLGCR,
	SYSTEMZ_INS_LLGF,
	SYSTEMZ_INS_LLGFAT,
	SYSTEMZ_INS_LLGFR,
	SYSTEMZ_INS_LLGFRL,
	SYSTEMZ_INS_LLGFSG,
	SYSTEMZ_INS_LLGH,
	SYSTEMZ_INS_LLGHR,
	SYSTEMZ_INS_LLGHRL,
	SYSTEMZ_INS_LLGT,
	SYSTEMZ_INS_LLGTAT,
	SYSTEMZ_INS_LLGTR,
	SYSTEMZ_INS_LLH,
	SYSTEMZ_INS_LLHH,
	SYSTEMZ_INS_LLHR,
	SYSTEMZ_INS_LLHRL,
	SYSTEMZ_INS_LLIHF,
	SYSTEMZ_INS_LLIHH,
	SYSTEMZ_INS_LLIHL,
	SYSTEMZ_INS_LLILF,
	SYSTEMZ_INS_LLILH,
	SYSTEMZ_INS_LLILL,
	SYSTEMZ_INS_LLZRGF,
	SYSTEMZ_INS_LM,
	SYSTEMZ_INS_LMD,
	SYSTEMZ_INS_LMG,
	SYSTEMZ_INS_LMH,
	SYSTEMZ_INS_LMY,
	SYSTEMZ_INS_LNDBR,
	SYSTEMZ_INS_LNDFR,
	SYSTEMZ_INS_LNDR,
	SYSTEMZ_INS_LNEBR,
	SYSTEMZ_INS_LNER,
	SYSTEMZ_INS_LNGFR,
	SYSTEMZ_INS_LNGR,
	SYSTEMZ_INS_LNR,
	SYSTEMZ_INS_LNXBR,
	SYSTEMZ_INS_LNXR,
	SYSTEMZ_INS_LOC,
	SYSTEMZ_INS_LOCE,
	SYSTEMZ_INS_LOCH,
	SYSTEMZ_INS_LOCHE,
	SYSTEMZ_INS_LOCL,
	SYSTEMZ_INS_LOCLE,
	SYSTEMZ_INS_LOCLH,
	SYSTEMZ_INS_LOCM,
	SYSTEMZ_INS_LOCNE,
	SYSTEMZ_INS_LOCNH,
	SYSTEMZ_INS_LOCNHE,
	SYSTEMZ_INS_LOCNL,
	SYSTEMZ_INS_LOCNLE,
	SYSTEMZ_INS_LOCNLH,
	SYSTEMZ_INS_LOCNM,
	SYSTEMZ_INS_LOCNO,
	SYSTEMZ_INS_LOCNP,
	SYSTEMZ_INS_LOCNZ,
	SYSTEMZ_INS_LOCO,
	SYSTEMZ_INS_LOCP,
	SYSTEMZ_INS_LOCZ,
	SYSTEMZ_INS_LOCFH,
	SYSTEMZ_INS_LOCFHE,
	SYSTEMZ_INS_LOCFHH,
	SYSTEMZ_INS_LOCFHHE,
	SYSTEMZ_INS_LOCFHL,
	SYSTEMZ_INS_LOCFHLE,
	SYSTEMZ_INS_LOCFHLH,
	SYSTEMZ_INS_LOCFHM,
	SYSTEMZ_INS_LOCFHNE,
	SYSTEMZ_INS_LOCFHNH,
	SYSTEMZ_INS_LOCFHNHE,
	SYSTEMZ_INS_LOCFHNL,
	SYSTEMZ_INS_LOCFHNLE,
	SYSTEMZ_INS_LOCFHNLH,
	SYSTEMZ_INS_LOCFHNM,
	SYSTEMZ_INS_LOCFHNO,
	SYSTEMZ_INS_LOCFHNP,
	SYSTEMZ_INS_LOCFHNZ,
	SYSTEMZ_INS_LOCFHO,
	SYSTEMZ_INS_LOCFHP,
	SYSTEMZ_INS_LOCFHZ,
	SYSTEMZ_INS_LOCFHR,
	SYSTEMZ_INS_LOCFHRE,
	SYSTEMZ_INS_LOCFHRH,
	SYSTEMZ_INS_LOCFHRHE,
	SYSTEMZ_INS_LOCFHRL,
	SYSTEMZ_INS_LOCFHRLE,
	SYSTEMZ_INS_LOCFHRLH,
	SYSTEMZ_INS_LOCFHRM,
	SYSTEMZ_INS_LOCFHRNE,
	SYSTEMZ_INS_LOCFHRNH,
	SYSTEMZ_INS_LOCFHRNHE,
	SYSTEMZ_INS_LOCFHRNL,
	SYSTEMZ_INS_LOCFHRNLE,
	SYSTEMZ_INS_LOCFHRNLH,
	SYSTEMZ_INS_LOCFHRNM,
	SYSTEMZ_INS_LOCFHRNO,
	SYSTEMZ_INS_LOCFHRNP,
	SYSTEMZ_INS_LOCFHRNZ,
	SYSTEMZ_INS_LOCFHRO,
	SYSTEMZ_INS_LOCFHRP,
	SYSTEMZ_INS_LOCFHRZ,
	SYSTEMZ_INS_LOCG,
	SYSTEMZ_INS_LOCGE,
	SYSTEMZ_INS_LOCGH,
	SYSTEMZ_INS_LOCGHE,
	SYSTEMZ_INS_LOCGL,
	SYSTEMZ_INS_LOCGLE,
	SYSTEMZ_INS_LOCGLH,
	SYSTEMZ_INS_LOCGM,
	SYSTEMZ_INS_LOCGNE,
	SYSTEMZ_INS_LOCGNH,
	SYSTEMZ_INS_LOCGNHE,
	SYSTEMZ_INS_LOCGNL,
	SYSTEMZ_INS_LOCGNLE,
	SYSTEMZ_INS_LOCGNLH,
	SYSTEMZ_INS_LOCGNM,
	SYSTEMZ_INS_LOCGNO,
	SYSTEMZ_INS_LOCGNP,
	SYSTEMZ_INS_LOCGNZ,
	SYSTEMZ_INS_LOCGO,
	SYSTEMZ_INS_LOCGP,
	SYSTEMZ_INS_LOCGZ,
	SYSTEMZ_INS_LOCGHI,
	SYSTEMZ_INS_LOCGHIE,
	SYSTEMZ_INS_LOCGHIH,
	SYSTEMZ_INS_LOCGHIHE,
	SYSTEMZ_INS_LOCGHIL,
	SYSTEMZ_INS_LOCGHILE,
	SYSTEMZ_INS_LOCGHILH,
	SYSTEMZ_INS_LOCGHIM,
	SYSTEMZ_INS_LOCGHINE,
	SYSTEMZ_INS_LOCGHINH,
	SYSTEMZ_INS_LOCGHINHE,
	SYSTEMZ_INS_LOCGHINL,
	SYSTEMZ_INS_LOCGHINLE,
	SYSTEMZ_INS_LOCGHINLH,
	SYSTEMZ_INS_LOCGHINM,
	SYSTEMZ_INS_LOCGHINO,
	SYSTEMZ_INS_LOCGHINP,
	SYSTEMZ_INS_LOCGHINZ,
	SYSTEMZ_INS_LOCGHIO,
	SYSTEMZ_INS_LOCGHIP,
	SYSTEMZ_INS_LOCGHIZ,
	SYSTEMZ_INS_LOCGR,
	SYSTEMZ_INS_LOCGRE,
	SYSTEMZ_INS_LOCGRH,
	SYSTEMZ_INS_LOCGRHE,
	SYSTEMZ_INS_LOCGRL,
	SYSTEMZ_INS_LOCGRLE,
	SYSTEMZ_INS_LOCGRLH,
	SYSTEMZ_INS_LOCGRM,
	SYSTEMZ_INS_LOCGRNE,
	SYSTEMZ_INS_LOCGRNH,
	SYSTEMZ_INS_LOCGRNHE,
	SYSTEMZ_INS_LOCGRNL,
	SYSTEMZ_INS_LOCGRNLE,
	SYSTEMZ_INS_LOCGRNLH,
	SYSTEMZ_INS_LOCGRNM,
	SYSTEMZ_INS_LOCGRNO,
	SYSTEMZ_INS_LOCGRNP,
	SYSTEMZ_INS_LOCGRNZ,
	SYSTEMZ_INS_LOCGRO,
	SYSTEMZ_INS_LOCGRP,
	SYSTEMZ_INS_LOCGRZ,
	SYSTEMZ_INS_LOCHHI,
	SYSTEMZ_INS_LOCHHIE,
	SYSTEMZ_INS_LOCHHIH,
	SYSTEMZ_INS_LOCHHIHE,
	SYSTEMZ_INS_LOCHHIL,
	SYSTEMZ_INS_LOCHHILE,
	SYSTEMZ_INS_LOCHHILH,
	SYSTEMZ_INS_LOCHHIM,
	SYSTEMZ_INS_LOCHHINE,
	SYSTEMZ_INS_LOCHHINH,
	SYSTEMZ_INS_LOCHHINHE,
	SYSTEMZ_INS_LOCHHINL,
	SYSTEMZ_INS_LOCHHINLE,
	SYSTEMZ_INS_LOCHHINLH,
	SYSTEMZ_INS_LOCHHINM,
	SYSTEMZ_INS_LOCHHINO,
	SYSTEMZ_INS_LOCHHINP,
	SYSTEMZ_INS_LOCHHINZ,
	SYSTEMZ_INS_LOCHHIO,
	SYSTEMZ_INS_LOCHHIP,
	SYSTEMZ_INS_LOCHHIZ,
	SYSTEMZ_INS_LOCHI,
	SYSTEMZ_INS_LOCHIE,
	SYSTEMZ_INS_LOCHIH,
	SYSTEMZ_INS_LOCHIHE,
	SYSTEMZ_INS_LOCHIL,
	SYSTEMZ_INS_LOCHILE,
	SYSTEMZ_INS_LOCHILH,
	SYSTEMZ_INS_LOCHIM,
	SYSTEMZ_INS_LOCHINE,
	SYSTEMZ_INS_LOCHINH,
	SYSTEMZ_INS_LOCHINHE,
	SYSTEMZ_INS_LOCHINL,
	SYSTEMZ_INS_LOCHINLE,
	SYSTEMZ_INS_LOCHINLH,
	SYSTEMZ_INS_LOCHINM,
	SYSTEMZ_INS_LOCHINO,
	SYSTEMZ_INS_LOCHINP,
	SYSTEMZ_INS_LOCHINZ,
	SYSTEMZ_INS_LOCHIO,
	SYSTEMZ_INS_LOCHIP,
	SYSTEMZ_INS_LOCHIZ,
	SYSTEMZ_INS_LOCR,
	SYSTEMZ_INS_LOCRE,
	SYSTEMZ_INS_LOCRH,
	SYSTEMZ_INS_LOCRHE,
	SYSTEMZ_INS_LOCRL,
	SYSTEMZ_INS_LOCRLE,
	SYSTEMZ_INS_LOCRLH,
	SYSTEMZ_INS_LOCRM,
	SYSTEMZ_INS_LOCRNE,
	SYSTEMZ_INS_LOCRNH,
	SYSTEMZ_INS_LOCRNHE,
	SYSTEMZ_INS_LOCRNL,
	SYSTEMZ_INS_LOCRNLE,
	SYSTEMZ_INS_LOCRNLH,
	SYSTEMZ_INS_LOCRNM,
	SYSTEMZ_INS_LOCRNO,
	SYSTEMZ_INS_LOCRNP,
	SYSTEMZ_INS_LOCRNZ,
	SYSTEMZ_INS_LOCRO,
	SYSTEMZ_INS_LOCRP,
	SYSTEMZ_INS_LOCRZ,
	SYSTEMZ_INS_LPCTL,
	SYSTEMZ_INS_LPD,
	SYSTEMZ_INS_LPDBR,
	SYSTEMZ_INS_LPDFR,
	SYSTEMZ_INS_LPDG,
	SYSTEMZ_INS_LPDR,
	SYSTEMZ_INS_LPEBR,
	SYSTEMZ_INS_LPER,
	SYSTEMZ_INS_LPGFR,
	SYSTEMZ_INS_LPGR,
	SYSTEMZ_INS_LPP,
	SYSTEMZ_INS_LPQ,
	SYSTEMZ_INS_LPR,
	SYSTEMZ_INS_LPSW,
	SYSTEMZ_INS_LPSWE,
	SYSTEMZ_INS_LPSWEY,
	SYSTEMZ_INS_LPTEA,
	SYSTEMZ_INS_LPXBR,
	SYSTEMZ_INS_LPXR,
	SYSTEMZ_INS_LR,
	SYSTEMZ_INS_LRA,
	SYSTEMZ_INS_LRAG,
	SYSTEMZ_INS_LRAY,
	SYSTEMZ_INS_LRDR,
	SYSTEMZ_INS_LRER,
	SYSTEMZ_INS_LRL,
	SYSTEMZ_INS_LRV,
	SYSTEMZ_INS_LRVG,
	SYSTEMZ_INS_LRVGR,
	SYSTEMZ_INS_LRVH,
	SYSTEMZ_INS_LRVR,
	SYSTEMZ_INS_LSCTL,
	SYSTEMZ_INS_LT,
	SYSTEMZ_INS_LTDBR,
	SYSTEMZ_INS_LTDR,
	SYSTEMZ_INS_LTDTR,
	SYSTEMZ_INS_LTEBR,
	SYSTEMZ_INS_LTER,
	SYSTEMZ_INS_LTG,
	SYSTEMZ_INS_LTGF,
	SYSTEMZ_INS_LTGFR,
	SYSTEMZ_INS_LTGR,
	SYSTEMZ_INS_LTR,
	SYSTEMZ_INS_LTXBR,
	SYSTEMZ_INS_LTXR,
	SYSTEMZ_INS_LTXTR,
	SYSTEMZ_INS_LURA,
	SYSTEMZ_INS_LURAG,
	SYSTEMZ_INS_LXD,
	SYSTEMZ_INS_LXDB,
	SYSTEMZ_INS_LXDBR,
	SYSTEMZ_INS_LXDR,
	SYSTEMZ_INS_LXDTR,
	SYSTEMZ_INS_LXE,
	SYSTEMZ_INS_LXEB,
	SYSTEMZ_INS_LXEBR,
	SYSTEMZ_INS_LXER,
	SYSTEMZ_INS_LXR,
	SYSTEMZ_INS_LY,
	SYSTEMZ_INS_LZDR,
	SYSTEMZ_INS_LZER,
	SYSTEMZ_INS_LZRF,
	SYSTEMZ_INS_LZRG,
	SYSTEMZ_INS_LZXR,
	SYSTEMZ_INS_M,
	SYSTEMZ_INS_MAD,
	SYSTEMZ_INS_MADB,
	SYSTEMZ_INS_MADBR,
	SYSTEMZ_INS_MADR,
	SYSTEMZ_INS_MAE,
	SYSTEMZ_INS_MAEB,
	SYSTEMZ_INS_MAEBR,
	SYSTEMZ_INS_MAER,
	SYSTEMZ_INS_MAY,
	SYSTEMZ_INS_MAYH,
	SYSTEMZ_INS_MAYHR,
	SYSTEMZ_INS_MAYL,
	SYSTEMZ_INS_MAYLR,
	SYSTEMZ_INS_MAYR,
	SYSTEMZ_INS_MC,
	SYSTEMZ_INS_MD,
	SYSTEMZ_INS_MDB,
	SYSTEMZ_INS_MDBR,
	SYSTEMZ_INS_MDE,
	SYSTEMZ_INS_MDEB,
	SYSTEMZ_INS_MDEBR,
	SYSTEMZ_INS_MDER,
	SYSTEMZ_INS_MDR,
	SYSTEMZ_INS_MDTR,
	SYSTEMZ_INS_MDTRA,
	SYSTEMZ_INS_ME,
	SYSTEMZ_INS_MEE,
	SYSTEMZ_INS_MEEB,
	SYSTEMZ_INS_MEEBR,
	SYSTEMZ_INS_MEER,
	SYSTEMZ_INS_MER,
	SYSTEMZ_INS_MFY,
	SYSTEMZ_INS_MG,
	SYSTEMZ_INS_MGH,
	SYSTEMZ_INS_MGHI,
	SYSTEMZ_INS_MGRK,
	SYSTEMZ_INS_MH,
	SYSTEMZ_INS_MHI,
	SYSTEMZ_INS_MHY,
	SYSTEMZ_INS_ML,
	SYSTEMZ_INS_MLG,
	SYSTEMZ_INS_MLGR,
	SYSTEMZ_INS_MLR,
	SYSTEMZ_INS_MP,
	SYSTEMZ_INS_MR,
	SYSTEMZ_INS_MS,
	SYSTEMZ_INS_MSC,
	SYSTEMZ_INS_MSCH,
	SYSTEMZ_INS_MSD,
	SYSTEMZ_INS_MSDB,
	SYSTEMZ_INS_MSDBR,
	SYSTEMZ_INS_MSDR,
	SYSTEMZ_INS_MSE,
	SYSTEMZ_INS_MSEB,
	SYSTEMZ_INS_MSEBR,
	SYSTEMZ_INS_MSER,
	SYSTEMZ_INS_MSFI,
	SYSTEMZ_INS_MSG,
	SYSTEMZ_INS_MSGC,
	SYSTEMZ_INS_MSGF,
	SYSTEMZ_INS_MSGFI,
	SYSTEMZ_INS_MSGFR,
	SYSTEMZ_INS_MSGR,
	SYSTEMZ_INS_MSGRKC,
	SYSTEMZ_INS_MSR,
	SYSTEMZ_INS_MSRKC,
	SYSTEMZ_INS_MSTA,
	SYSTEMZ_INS_MSY,
	SYSTEMZ_INS_MVC,
	SYSTEMZ_INS_MVCDK,
	SYSTEMZ_INS_MVCIN,
	SYSTEMZ_INS_MVCK,
	SYSTEMZ_INS_MVCL,
	SYSTEMZ_INS_MVCLE,
	SYSTEMZ_INS_MVCLU,
	SYSTEMZ_INS_MVCOS,
	SYSTEMZ_INS_MVCP,
	SYSTEMZ_INS_MVCRL,
	SYSTEMZ_INS_MVCS,
	SYSTEMZ_INS_MVCSK,
	SYSTEMZ_INS_MVGHI,
	SYSTEMZ_INS_MVHHI,
	SYSTEMZ_INS_MVHI,
	SYSTEMZ_INS_MVI,
	SYSTEMZ_INS_MVIY,
	SYSTEMZ_INS_MVN,
	SYSTEMZ_INS_MVO,
	SYSTEMZ_INS_MVPG,
	SYSTEMZ_INS_MVST,
	SYSTEMZ_INS_MVZ,
	SYSTEMZ_INS_MXBR,
	SYSTEMZ_INS_MXD,
	SYSTEMZ_INS_MXDB,
	SYSTEMZ_INS_MXDBR,
	SYSTEMZ_INS_MXDR,
	SYSTEMZ_INS_MXR,
	SYSTEMZ_INS_MXTR,
	SYSTEMZ_INS_MXTRA,
	SYSTEMZ_INS_MY,
	SYSTEMZ_INS_MYH,
	SYSTEMZ_INS_MYHR,
	SYSTEMZ_INS_MYL,
	SYSTEMZ_INS_MYLR,
	SYSTEMZ_INS_MYR,
	SYSTEMZ_INS_N,
	SYSTEMZ_INS_NC,
	SYSTEMZ_INS_NCGRK,
	SYSTEMZ_INS_NCRK,
	SYSTEMZ_INS_NG,
	SYSTEMZ_INS_NGR,
	SYSTEMZ_INS_NGRK,
	SYSTEMZ_INS_NI,
	SYSTEMZ_INS_NIAI,
	SYSTEMZ_INS_NIHF,
	SYSTEMZ_INS_NIHH,
	SYSTEMZ_INS_NIHL,
	SYSTEMZ_INS_NILF,
	SYSTEMZ_INS_NILH,
	SYSTEMZ_INS_NILL,
	SYSTEMZ_INS_NIY,
	SYSTEMZ_INS_NNGRK,
	SYSTEMZ_INS_NNPA,
	SYSTEMZ_INS_NNRK,
	SYSTEMZ_INS_NOGRK,
	SYSTEMZ_INS_NOP,
	SYSTEMZ_INS_NORK,
	SYSTEMZ_INS_NR,
	SYSTEMZ_INS_NRK,
	SYSTEMZ_INS_NTSTG,
	SYSTEMZ_INS_NXGRK,
	SYSTEMZ_INS_NXRK,
	SYSTEMZ_INS_NY,
	SYSTEMZ_INS_O,
	SYSTEMZ_INS_OC,
	SYSTEMZ_INS_OCGRK,
	SYSTEMZ_INS_OCRK,
	SYSTEMZ_INS_OG,
	SYSTEMZ_INS_OGR,
	SYSTEMZ_INS_OGRK,
	SYSTEMZ_INS_OI,
	SYSTEMZ_INS_OIHF,
	SYSTEMZ_INS_OIHH,
	SYSTEMZ_INS_OIHL,
	SYSTEMZ_INS_OILF,
	SYSTEMZ_INS_OILH,
	SYSTEMZ_INS_OILL,
	SYSTEMZ_INS_OIY,
	SYSTEMZ_INS_OR,
	SYSTEMZ_INS_ORK,
	SYSTEMZ_INS_OY,
	SYSTEMZ_INS_PACK,
	SYSTEMZ_INS_PALB,
	SYSTEMZ_INS_PC,
	SYSTEMZ_INS_PCC,
	SYSTEMZ_INS_PCKMO,
	SYSTEMZ_INS_PFD,
	SYSTEMZ_INS_PFDRL,
	SYSTEMZ_INS_PFMF,
	SYSTEMZ_INS_PFPO,
	SYSTEMZ_INS_PGIN,
	SYSTEMZ_INS_PGOUT,
	SYSTEMZ_INS_PKA,
	SYSTEMZ_INS_PKU,
	SYSTEMZ_INS_PLO,
	SYSTEMZ_INS_POPCNT,
	SYSTEMZ_INS_PPA,
	SYSTEMZ_INS_PPNO,
	SYSTEMZ_INS_PR,
	SYSTEMZ_INS_PRNO,
	SYSTEMZ_INS_PT,
	SYSTEMZ_INS_PTF,
	SYSTEMZ_INS_PTFF,
	SYSTEMZ_INS_PTI,
	SYSTEMZ_INS_PTLB,
	SYSTEMZ_INS_QADTR,
	SYSTEMZ_INS_QAXTR,
	SYSTEMZ_INS_QCTRI,
	SYSTEMZ_INS_QPACI,
	SYSTEMZ_INS_QSI,
	SYSTEMZ_INS_RCHP,
	SYSTEMZ_INS_RDP,
	SYSTEMZ_INS_RISBG,
	SYSTEMZ_INS_RISBGN,
	SYSTEMZ_INS_RISBHG,
	SYSTEMZ_INS_RISBLG,
	SYSTEMZ_INS_RLL,
	SYSTEMZ_INS_RLLG,
	SYSTEMZ_INS_RNSBG,
	SYSTEMZ_INS_ROSBG,
	SYSTEMZ_INS_RP,
	SYSTEMZ_INS_RRBE,
	SYSTEMZ_INS_RRBM,
	SYSTEMZ_INS_RRDTR,
	SYSTEMZ_INS_RRXTR,
	SYSTEMZ_INS_RSCH,
	SYSTEMZ_INS_RXSBG,
	SYSTEMZ_INS_S,
	SYSTEMZ_INS_SAC,
	SYSTEMZ_INS_SACF,
	SYSTEMZ_INS_SAL,
	SYSTEMZ_INS_SAM24,
	SYSTEMZ_INS_SAM31,
	SYSTEMZ_INS_SAM64,
	SYSTEMZ_INS_SAR,
	SYSTEMZ_INS_SCCTR,
	SYSTEMZ_INS_SCHM,
	SYSTEMZ_INS_SCK,
	SYSTEMZ_INS_SCKC,
	SYSTEMZ_INS_SCKPF,
	SYSTEMZ_INS_SD,
	SYSTEMZ_INS_SDB,
	SYSTEMZ_INS_SDBR,
	SYSTEMZ_INS_SDR,
	SYSTEMZ_INS_SDTR,
	SYSTEMZ_INS_SDTRA,
	SYSTEMZ_INS_SE,
	SYSTEMZ_INS_SEB,
	SYSTEMZ_INS_SEBR,
	SYSTEMZ_INS_SELFHR,
	SYSTEMZ_INS_SELFHRE,
	SYSTEMZ_INS_SELFHRH,
	SYSTEMZ_INS_SELFHRHE,
	SYSTEMZ_INS_SELFHRL,
	SYSTEMZ_INS_SELFHRLE,
	SYSTEMZ_INS_SELFHRLH,
	SYSTEMZ_INS_SELFHRM,
	SYSTEMZ_INS_SELFHRNE,
	SYSTEMZ_INS_SELFHRNH,
	SYSTEMZ_INS_SELFHRNHE,
	SYSTEMZ_INS_SELFHRNL,
	SYSTEMZ_INS_SELFHRNLE,
	SYSTEMZ_INS_SELFHRNLH,
	SYSTEMZ_INS_SELFHRNM,
	SYSTEMZ_INS_SELFHRNO,
	SYSTEMZ_INS_SELFHRNP,
	SYSTEMZ_INS_SELFHRNZ,
	SYSTEMZ_INS_SELFHRO,
	SYSTEMZ_INS_SELFHRP,
	SYSTEMZ_INS_SELFHRZ,
	SYSTEMZ_INS_SELGR,
	SYSTEMZ_INS_SELGRE,
	SYSTEMZ_INS_SELGRH,
	SYSTEMZ_INS_SELGRHE,
	SYSTEMZ_INS_SELGRL,
	SYSTEMZ_INS_SELGRLE,
	SYSTEMZ_INS_SELGRLH,
	SYSTEMZ_INS_SELGRM,
	SYSTEMZ_INS_SELGRNE,
	SYSTEMZ_INS_SELGRNH,
	SYSTEMZ_INS_SELGRNHE,
	SYSTEMZ_INS_SELGRNL,
	SYSTEMZ_INS_SELGRNLE,
	SYSTEMZ_INS_SELGRNLH,
	SYSTEMZ_INS_SELGRNM,
	SYSTEMZ_INS_SELGRNO,
	SYSTEMZ_INS_SELGRNP,
	SYSTEMZ_INS_SELGRNZ,
	SYSTEMZ_INS_SELGRO,
	SYSTEMZ_INS_SELGRP,
	SYSTEMZ_INS_SELGRZ,
	SYSTEMZ_INS_SELR,
	SYSTEMZ_INS_SELRE,
	SYSTEMZ_INS_SELRH,
	SYSTEMZ_INS_SELRHE,
	SYSTEMZ_INS_SELRL,
	SYSTEMZ_INS_SELRLE,
	SYSTEMZ_INS_SELRLH,
	SYSTEMZ_INS_SELRM,
	SYSTEMZ_INS_SELRNE,
	SYSTEMZ_INS_SELRNH,
	SYSTEMZ_INS_SELRNHE,
	SYSTEMZ_INS_SELRNL,
	SYSTEMZ_INS_SELRNLE,
	SYSTEMZ_INS_SELRNLH,
	SYSTEMZ_INS_SELRNM,
	SYSTEMZ_INS_SELRNO,
	SYSTEMZ_INS_SELRNP,
	SYSTEMZ_INS_SELRNZ,
	SYSTEMZ_INS_SELRO,
	SYSTEMZ_INS_SELRP,
	SYSTEMZ_INS_SELRZ,
	SYSTEMZ_INS_SER,
	SYSTEMZ_INS_SFASR,
	SYSTEMZ_INS_SFPC,
	SYSTEMZ_INS_SG,
	SYSTEMZ_INS_SGF,
	SYSTEMZ_INS_SGFR,
	SYSTEMZ_INS_SGH,
	SYSTEMZ_INS_SGR,
	SYSTEMZ_INS_SGRK,
	SYSTEMZ_INS_SH,
	SYSTEMZ_INS_SHHHR,
	SYSTEMZ_INS_SHHLR,
	SYSTEMZ_INS_SHY,
	SYSTEMZ_INS_SIE,
	SYSTEMZ_INS_SIGA,
	SYSTEMZ_INS_SIGP,
	SYSTEMZ_INS_SL,
	SYSTEMZ_INS_SLA,
	SYSTEMZ_INS_SLAG,
	SYSTEMZ_INS_SLAK,
	SYSTEMZ_INS_SLB,
	SYSTEMZ_INS_SLBG,
	SYSTEMZ_INS_SLBGR,
	SYSTEMZ_INS_SLBR,
	SYSTEMZ_INS_SLDA,
	SYSTEMZ_INS_SLDL,
	SYSTEMZ_INS_SLDT,
	SYSTEMZ_INS_SLFI,
	SYSTEMZ_INS_SLG,
	SYSTEMZ_INS_SLGF,
	SYSTEMZ_INS_SLGFI,
	SYSTEMZ_INS_SLGFR,
	SYSTEMZ_INS_SLGR,
	SYSTEMZ_INS_SLGRK,
	SYSTEMZ_INS_SLHHHR,
	SYSTEMZ_INS_SLHHLR,
	SYSTEMZ_INS_SLL,
	SYSTEMZ_INS_SLLG,
	SYSTEMZ_INS_SLLK,
	SYSTEMZ_INS_SLR,
	SYSTEMZ_INS_SLRK,
	SYSTEMZ_INS_SLXT,
	SYSTEMZ_INS_SLY,
	SYSTEMZ_INS_SORTL,
	SYSTEMZ_INS_SP,
	SYSTEMZ_INS_SPCTR,
	SYSTEMZ_INS_SPKA,
	SYSTEMZ_INS_SPM,
	SYSTEMZ_INS_SPT,
	SYSTEMZ_INS_SPX,
	SYSTEMZ_INS_SQD,
	SYSTEMZ_INS_SQDB,
	SYSTEMZ_INS_SQDBR,
	SYSTEMZ_INS_SQDR,
	SYSTEMZ_INS_SQE,
	SYSTEMZ_INS_SQEB,
	SYSTEMZ_INS_SQEBR,
	SYSTEMZ_INS_SQER,
	SYSTEMZ_INS_SQXBR,
	SYSTEMZ_INS_SQXR,
	SYSTEMZ_INS_SR,
	SYSTEMZ_INS_SRA,
	SYSTEMZ_INS_SRAG,
	SYSTEMZ_INS_SRAK,
	SYSTEMZ_INS_SRDA,
	SYSTEMZ_INS_SRDL,
	SYSTEMZ_INS_SRDT,
	SYSTEMZ_INS_SRK,
	SYSTEMZ_INS_SRL,
	SYSTEMZ_INS_SRLG,
	SYSTEMZ_INS_SRLK,
	SYSTEMZ_INS_SRNM,
	SYSTEMZ_INS_SRNMB,
	SYSTEMZ_INS_SRNMT,
	SYSTEMZ_INS_SRP,
	SYSTEMZ_INS_SRST,
	SYSTEMZ_INS_SRSTU,
	SYSTEMZ_INS_SRXT,
	SYSTEMZ_INS_SSAIR,
	SYSTEMZ_INS_SSAR,
	SYSTEMZ_INS_SSCH,
	SYSTEMZ_INS_SSKE,
	SYSTEMZ_INS_SSM,
	SYSTEMZ_INS_ST,
	SYSTEMZ_INS_STAM,
	SYSTEMZ_INS_STAMY,
	SYSTEMZ_INS_STAP,
	SYSTEMZ_INS_STBEAR,
	SYSTEMZ_INS_STC,
	SYSTEMZ_INS_STCH,
	SYSTEMZ_INS_STCK,
	SYSTEMZ_INS_STCKC,
	SYSTEMZ_INS_STCKE,
	SYSTEMZ_INS_STCKF,
	SYSTEMZ_INS_STCM,
	SYSTEMZ_INS_STCMH,
	SYSTEMZ_INS_STCMY,
	SYSTEMZ_INS_STCPS,
	SYSTEMZ_INS_STCRW,
	SYSTEMZ_INS_STCTG,
	SYSTEMZ_INS_STCTL,
	SYSTEMZ_INS_STCY,
	SYSTEMZ_INS_STD,
	SYSTEMZ_INS_STDY,
	SYSTEMZ_INS_STE,
	SYSTEMZ_INS_STEY,
	SYSTEMZ_INS_STFH,
	SYSTEMZ_INS_STFL,
	SYSTEMZ_INS_STFLE,
	SYSTEMZ_INS_STFPC,
	SYSTEMZ_INS_STG,
	SYSTEMZ_INS_STGRL,
	SYSTEMZ_INS_STGSC,
	SYSTEMZ_INS_STH,
	SYSTEMZ_INS_STHH,
	SYSTEMZ_INS_STHRL,
	SYSTEMZ_INS_STHY,
	SYSTEMZ_INS_STIDP,
	SYSTEMZ_INS_STM,
	SYSTEMZ_INS_STMG,
	SYSTEMZ_INS_STMH,
	SYSTEMZ_INS_STMY,
	SYSTEMZ_INS_STNSM,
	SYSTEMZ_INS_STOC,
	SYSTEMZ_INS_STOCE,
	SYSTEMZ_INS_STOCH,
	SYSTEMZ_INS_STOCHE,
	SYSTEMZ_INS_STOCL,
	SYSTEMZ_INS_STOCLE,
	SYSTEMZ_INS_STOCLH,
	SYSTEMZ_INS_STOCM,
	SYSTEMZ_INS_STOCNE,
	SYSTEMZ_INS_STOCNH,
	SYSTEMZ_INS_STOCNHE,
	SYSTEMZ_INS_STOCNL,
	SYSTEMZ_INS_STOCNLE,
	SYSTEMZ_INS_STOCNLH,
	SYSTEMZ_INS_STOCNM,
	SYSTEMZ_INS_STOCNO,
	SYSTEMZ_INS_STOCNP,
	SYSTEMZ_INS_STOCNZ,
	SYSTEMZ_INS_STOCO,
	SYSTEMZ_INS_STOCP,
	SYSTEMZ_INS_STOCZ,
	SYSTEMZ_INS_STOCFH,
	SYSTEMZ_INS_STOCFHE,
	SYSTEMZ_INS_STOCFHH,
	SYSTEMZ_INS_STOCFHHE,
	SYSTEMZ_INS_STOCFHL,
	SYSTEMZ_INS_STOCFHLE,
	SYSTEMZ_INS_STOCFHLH,
	SYSTEMZ_INS_STOCFHM,
	SYSTEMZ_INS_STOCFHNE,
	SYSTEMZ_INS_STOCFHNH,
	SYSTEMZ_INS_STOCFHNHE,
	SYSTEMZ_INS_STOCFHNL,
	SYSTEMZ_INS_STOCFHNLE,
	SYSTEMZ_INS_STOCFHNLH,
	SYSTEMZ_INS_STOCFHNM,
	SYSTEMZ_INS_STOCFHNO,
	SYSTEMZ_INS_STOCFHNP,
	SYSTEMZ_INS_STOCFHNZ,
	SYSTEMZ_INS_STOCFHO,
	SYSTEMZ_INS_STOCFHP,
	SYSTEMZ_INS_STOCFHZ,
	SYSTEMZ_INS_STOCG,
	SYSTEMZ_INS_STOCGE,
	SYSTEMZ_INS_STOCGH,
	SYSTEMZ_INS_STOCGHE,
	SYSTEMZ_INS_STOCGL,
	SYSTEMZ_INS_STOCGLE,
	SYSTEMZ_INS_STOCGLH,
	SYSTEMZ_INS_STOCGM,
	SYSTEMZ_INS_STOCGNE,
	SYSTEMZ_INS_STOCGNH,
	SYSTEMZ_INS_STOCGNHE,
	SYSTEMZ_INS_STOCGNL,
	SYSTEMZ_INS_STOCGNLE,
	SYSTEMZ_INS_STOCGNLH,
	SYSTEMZ_INS_STOCGNM,
	SYSTEMZ_INS_STOCGNO,
	SYSTEMZ_INS_STOCGNP,
	SYSTEMZ_INS_STOCGNZ,
	SYSTEMZ_INS_STOCGO,
	SYSTEMZ_INS_STOCGP,
	SYSTEMZ_INS_STOCGZ,
	SYSTEMZ_INS_STOSM,
	SYSTEMZ_INS_STPQ,
	SYSTEMZ_INS_STPT,
	SYSTEMZ_INS_STPX,
	SYSTEMZ_INS_STRAG,
	SYSTEMZ_INS_STRL,
	SYSTEMZ_INS_STRV,
	SYSTEMZ_INS_STRVG,
	SYSTEMZ_INS_STRVH,
	SYSTEMZ_INS_STSCH,
	SYSTEMZ_INS_STSI,
	SYSTEMZ_INS_STURA,
	SYSTEMZ_INS_STURG,
	SYSTEMZ_INS_STY,
	SYSTEMZ_INS_SU,
	SYSTEMZ_INS_SUR,
	SYSTEMZ_INS_SVC,
	SYSTEMZ_INS_SW,
	SYSTEMZ_INS_SWR,
	SYSTEMZ_INS_SXBR,
	SYSTEMZ_INS_SXR,
	SYSTEMZ_INS_SXTR,
	SYSTEMZ_INS_SXTRA,
	SYSTEMZ_INS_SY,
	SYSTEMZ_INS_TABORT,
	SYSTEMZ_INS_TAM,
	SYSTEMZ_INS_TAR,
	SYSTEMZ_INS_TB,
	SYSTEMZ_INS_TBDR,
	SYSTEMZ_INS_TBEDR,
	SYSTEMZ_INS_TBEGIN,
	SYSTEMZ_INS_TBEGINC,
	SYSTEMZ_INS_TCDB,
	SYSTEMZ_INS_TCEB,
	SYSTEMZ_INS_TCXB,
	SYSTEMZ_INS_TDCDT,
	SYSTEMZ_INS_TDCET,
	SYSTEMZ_INS_TDCXT,
	SYSTEMZ_INS_TDGDT,
	SYSTEMZ_INS_TDGET,
	SYSTEMZ_INS_TDGXT,
	SYSTEMZ_INS_TEND,
	SYSTEMZ_INS_THDER,
	SYSTEMZ_INS_THDR,
	SYSTEMZ_INS_TM,
	SYSTEMZ_INS_TMHH,
	SYSTEMZ_INS_TMHL,
	SYSTEMZ_INS_TMLH,
	SYSTEMZ_INS_TMLL,
	SYSTEMZ_INS_TMY,
	SYSTEMZ_INS_TP,
	SYSTEMZ_INS_TPI,
	SYSTEMZ_INS_TPROT,
	SYSTEMZ_INS_TR,
	SYSTEMZ_INS_TRACE,
	SYSTEMZ_INS_TRACG,
	SYSTEMZ_INS_TRAP2,
	SYSTEMZ_INS_TRAP4,
	SYSTEMZ_INS_TRE,
	SYSTEMZ_INS_TROO,
	SYSTEMZ_INS_TROT,
	SYSTEMZ_INS_TRT,
	SYSTEMZ_INS_TRTE,
	SYSTEMZ_INS_TRTO,
	SYSTEMZ_INS_TRTR,
	SYSTEMZ_INS_TRTRE,
	SYSTEMZ_INS_TRTT,
	SYSTEMZ_INS_TS,
	SYSTEMZ_INS_TSCH,
	SYSTEMZ_INS_UNPK,
	SYSTEMZ_INS_UNPKA,
	SYSTEMZ_INS_UNPKU,
	SYSTEMZ_INS_UPT,
	SYSTEMZ_INS_VA,
	SYSTEMZ_INS_VAB,
	SYSTEMZ_INS_VAC,
	SYSTEMZ_INS_VACC,
	SYSTEMZ_INS_VACCB,
	SYSTEMZ_INS_VACCC,
	SYSTEMZ_INS_VACCCQ,
	SYSTEMZ_INS_VACCF,
	SYSTEMZ_INS_VACCG,
	SYSTEMZ_INS_VACCH,
	SYSTEMZ_INS_VACCQ,
	SYSTEMZ_INS_VACQ,
	SYSTEMZ_INS_VAF,
	SYSTEMZ_INS_VAG,
	SYSTEMZ_INS_VAH,
	SYSTEMZ_INS_VAP,
	SYSTEMZ_INS_VAQ,
	SYSTEMZ_INS_VAVG,
	SYSTEMZ_INS_VAVGB,
	SYSTEMZ_INS_VAVGF,
	SYSTEMZ_INS_VAVGG,
	SYSTEMZ_INS_VAVGH,
	SYSTEMZ_INS_VAVGL,
	SYSTEMZ_INS_VAVGLB,
	SYSTEMZ_INS_VAVGLF,
	SYSTEMZ_INS_VAVGLG,
	SYSTEMZ_INS_VAVGLH,
	SYSTEMZ_INS_VBPERM,
	SYSTEMZ_INS_VCDG,
	SYSTEMZ_INS_VCDGB,
	SYSTEMZ_INS_VCDLG,
	SYSTEMZ_INS_VCDLGB,
	SYSTEMZ_INS_VCEFB,
	SYSTEMZ_INS_VCELFB,
	SYSTEMZ_INS_VCEQ,
	SYSTEMZ_INS_VCEQB,
	SYSTEMZ_INS_VCEQBS,
	SYSTEMZ_INS_VCEQF,
	SYSTEMZ_INS_VCEQFS,
	SYSTEMZ_INS_VCEQG,
	SYSTEMZ_INS_VCEQGS,
	SYSTEMZ_INS_VCEQH,
	SYSTEMZ_INS_VCEQHS,
	SYSTEMZ_INS_VCFEB,
	SYSTEMZ_INS_VCFN,
	SYSTEMZ_INS_VCFPL,
	SYSTEMZ_INS_VCFPS,
	SYSTEMZ_INS_VCGD,
	SYSTEMZ_INS_VCGDB,
	SYSTEMZ_INS_VCH,
	SYSTEMZ_INS_VCHB,
	SYSTEMZ_INS_VCHBS,
	SYSTEMZ_INS_VCHF,
	SYSTEMZ_INS_VCHFS,
	SYSTEMZ_INS_VCHG,
	SYSTEMZ_INS_VCHGS,
	SYSTEMZ_INS_VCHH,
	SYSTEMZ_INS_VCHHS,
	SYSTEMZ_INS_VCHL,
	SYSTEMZ_INS_VCHLB,
	SYSTEMZ_INS_VCHLBS,
	SYSTEMZ_INS_VCHLF,
	SYSTEMZ_INS_VCHLFS,
	SYSTEMZ_INS_VCHLG,
	SYSTEMZ_INS_VCHLGS,
	SYSTEMZ_INS_VCHLH,
	SYSTEMZ_INS_VCHLHS,
	SYSTEMZ_INS_VCKSM,
	SYSTEMZ_INS_VCLFEB,
	SYSTEMZ_INS_VCLFNH,
	SYSTEMZ_INS_VCLFNL,
	SYSTEMZ_INS_VCLFP,
	SYSTEMZ_INS_VCLGD,
	SYSTEMZ_INS_VCLGDB,
	SYSTEMZ_INS_VCLZ,
	SYSTEMZ_INS_VCLZB,
	SYSTEMZ_INS_VCLZDP,
	SYSTEMZ_INS_VCLZF,
	SYSTEMZ_INS_VCLZG,
	SYSTEMZ_INS_VCLZH,
	SYSTEMZ_INS_VCNF,
	SYSTEMZ_INS_VCP,
	SYSTEMZ_INS_VCRNF,
	SYSTEMZ_INS_VCSFP,
	SYSTEMZ_INS_VCSPH,
	SYSTEMZ_INS_VCTZ,
	SYSTEMZ_INS_VCTZB,
	SYSTEMZ_INS_VCTZF,
	SYSTEMZ_INS_VCTZG,
	SYSTEMZ_INS_VCTZH,
	SYSTEMZ_INS_VCVB,
	SYSTEMZ_INS_VCVBG,
	SYSTEMZ_INS_VCVD,
	SYSTEMZ_INS_VCVDG,
	SYSTEMZ_INS_VDP,
	SYSTEMZ_INS_VEC,
	SYSTEMZ_INS_VECB,
	SYSTEMZ_INS_VECF,
	SYSTEMZ_INS_VECG,
	SYSTEMZ_INS_VECH,
	SYSTEMZ_INS_VECL,
	SYSTEMZ_INS_VECLB,
	SYSTEMZ_INS_VECLF,
	SYSTEMZ_INS_VECLG,
	SYSTEMZ_INS_VECLH,
	SYSTEMZ_INS_VERIM,
	SYSTEMZ_INS_VERIMB,
	SYSTEMZ_INS_VERIMF,
	SYSTEMZ_INS_VERIMG,
	SYSTEMZ_INS_VERIMH,
	SYSTEMZ_INS_VERLL,
	SYSTEMZ_INS_VERLLB,
	SYSTEMZ_INS_VERLLF,
	SYSTEMZ_INS_VERLLG,
	SYSTEMZ_INS_VERLLH,
	SYSTEMZ_INS_VERLLV,
	SYSTEMZ_INS_VERLLVB,
	SYSTEMZ_INS_VERLLVF,
	SYSTEMZ_INS_VERLLVG,
	SYSTEMZ_INS_VERLLVH,
	SYSTEMZ_INS_VESL,
	SYSTEMZ_INS_VESLB,
	SYSTEMZ_INS_VESLF,
	SYSTEMZ_INS_VESLG,
	SYSTEMZ_INS_VESLH,
	SYSTEMZ_INS_VESLV,
	SYSTEMZ_INS_VESLVB,
	SYSTEMZ_INS_VESLVF,
	SYSTEMZ_INS_VESLVG,
	SYSTEMZ_INS_VESLVH,
	SYSTEMZ_INS_VESRA,
	SYSTEMZ_INS_VESRAB,
	SYSTEMZ_INS_VESRAF,
	SYSTEMZ_INS_VESRAG,
	SYSTEMZ_INS_VESRAH,
	SYSTEMZ_INS_VESRAV,
	SYSTEMZ_INS_VESRAVB,
	SYSTEMZ_INS_VESRAVF,
	SYSTEMZ_INS_VESRAVG,
	SYSTEMZ_INS_VESRAVH,
	SYSTEMZ_INS_VESRL,
	SYSTEMZ_INS_VESRLB,
	SYSTEMZ_INS_VESRLF,
	SYSTEMZ_INS_VESRLG,
	SYSTEMZ_INS_VESRLH,
	SYSTEMZ_INS_VESRLV,
	SYSTEMZ_INS_VESRLVB,
	SYSTEMZ_INS_VESRLVF,
	SYSTEMZ_INS_VESRLVG,
	SYSTEMZ_INS_VESRLVH,
	SYSTEMZ_INS_VFA,
	SYSTEMZ_INS_VFADB,
	SYSTEMZ_INS_VFAE,
	SYSTEMZ_INS_VFAEB,
	SYSTEMZ_INS_VFAEBS,
	SYSTEMZ_INS_VFAEF,
	SYSTEMZ_INS_VFAEFS,
	SYSTEMZ_INS_VFAEH,
	SYSTEMZ_INS_VFAEHS,
	SYSTEMZ_INS_VFAEZB,
	SYSTEMZ_INS_VFAEZBS,
	SYSTEMZ_INS_VFAEZF,
	SYSTEMZ_INS_VFAEZFS,
	SYSTEMZ_INS_VFAEZH,
	SYSTEMZ_INS_VFAEZHS,
	SYSTEMZ_INS_VFASB,
	SYSTEMZ_INS_VFCE,
	SYSTEMZ_INS_VFCEDB,
	SYSTEMZ_INS_VFCEDBS,
	SYSTEMZ_INS_VFCESB,
	SYSTEMZ_INS_VFCESBS,
	SYSTEMZ_INS_VFCH,
	SYSTEMZ_INS_VFCHDB,
	SYSTEMZ_INS_VFCHDBS,
	SYSTEMZ_INS_VFCHE,
	SYSTEMZ_INS_VFCHEDB,
	SYSTEMZ_INS_VFCHEDBS,
	SYSTEMZ_INS_VFCHESB,
	SYSTEMZ_INS_VFCHESBS,
	SYSTEMZ_INS_VFCHSB,
	SYSTEMZ_INS_VFCHSBS,
	SYSTEMZ_INS_VFD,
	SYSTEMZ_INS_VFDDB,
	SYSTEMZ_INS_VFDSB,
	SYSTEMZ_INS_VFEE,
	SYSTEMZ_INS_VFEEB,
	SYSTEMZ_INS_VFEEBS,
	SYSTEMZ_INS_VFEEF,
	SYSTEMZ_INS_VFEEFS,
	SYSTEMZ_INS_VFEEH,
	SYSTEMZ_INS_VFEEHS,
	SYSTEMZ_INS_VFEEZB,
	SYSTEMZ_INS_VFEEZBS,
	SYSTEMZ_INS_VFEEZF,
	SYSTEMZ_INS_VFEEZFS,
	SYSTEMZ_INS_VFEEZH,
	SYSTEMZ_INS_VFEEZHS,
	SYSTEMZ_INS_VFENE,
	SYSTEMZ_INS_VFENEB,
	SYSTEMZ_INS_VFENEBS,
	SYSTEMZ_INS_VFENEF,
	SYSTEMZ_INS_VFENEFS,
	SYSTEMZ_INS_VFENEH,
	SYSTEMZ_INS_VFENEHS,
	SYSTEMZ_INS_VFENEZB,
	SYSTEMZ_INS_VFENEZBS,
	SYSTEMZ_INS_VFENEZF,
	SYSTEMZ_INS_VFENEZFS,
	SYSTEMZ_INS_VFENEZH,
	SYSTEMZ_INS_VFENEZHS,
	SYSTEMZ_INS_VFI,
	SYSTEMZ_INS_VFIDB,
	SYSTEMZ_INS_VFISB,
	SYSTEMZ_INS_VFKEDB,
	SYSTEMZ_INS_VFKEDBS,
	SYSTEMZ_INS_VFKESB,
	SYSTEMZ_INS_VFKESBS,
	SYSTEMZ_INS_VFKHDB,
	SYSTEMZ_INS_VFKHDBS,
	SYSTEMZ_INS_VFKHEDB,
	SYSTEMZ_INS_VFKHEDBS,
	SYSTEMZ_INS_VFKHESB,
	SYSTEMZ_INS_VFKHESBS,
	SYSTEMZ_INS_VFKHSB,
	SYSTEMZ_INS_VFKHSBS,
	SYSTEMZ_INS_VFLCDB,
	SYSTEMZ_INS_VFLCSB,
	SYSTEMZ_INS_VFLL,
	SYSTEMZ_INS_VFLLS,
	SYSTEMZ_INS_VFLNDB,
	SYSTEMZ_INS_VFLNSB,
	SYSTEMZ_INS_VFLPDB,
	SYSTEMZ_INS_VFLPSB,
	SYSTEMZ_INS_VFLR,
	SYSTEMZ_INS_VFLRD,
	SYSTEMZ_INS_VFM,
	SYSTEMZ_INS_VFMA,
	SYSTEMZ_INS_VFMADB,
	SYSTEMZ_INS_VFMASB,
	SYSTEMZ_INS_VFMAX,
	SYSTEMZ_INS_VFMAXDB,
	SYSTEMZ_INS_VFMAXSB,
	SYSTEMZ_INS_VFMDB,
	SYSTEMZ_INS_VFMIN,
	SYSTEMZ_INS_VFMINDB,
	SYSTEMZ_INS_VFMINSB,
	SYSTEMZ_INS_VFMS,
	SYSTEMZ_INS_VFMSB,
	SYSTEMZ_INS_VFMSDB,
	SYSTEMZ_INS_VFMSSB,
	SYSTEMZ_INS_VFNMA,
	SYSTEMZ_INS_VFNMADB,
	SYSTEMZ_INS_VFNMASB,
	SYSTEMZ_INS_VFNMS,
	SYSTEMZ_INS_VFNMSDB,
	SYSTEMZ_INS_VFNMSSB,
	SYSTEMZ_INS_VFPSO,
	SYSTEMZ_INS_VFPSODB,
	SYSTEMZ_INS_VFPSOSB,
	SYSTEMZ_INS_VFS,
	SYSTEMZ_INS_VFSDB,
	SYSTEMZ_INS_VFSQ,
	SYSTEMZ_INS_VFSQDB,
	SYSTEMZ_INS_VFSQSB,
	SYSTEMZ_INS_VFSSB,
	SYSTEMZ_INS_VFTCI,
	SYSTEMZ_INS_VFTCIDB,
	SYSTEMZ_INS_VFTCISB,
	SYSTEMZ_INS_VGBM,
	SYSTEMZ_INS_VGEF,
	SYSTEMZ_INS_VGEG,
	SYSTEMZ_INS_VGFM,
	SYSTEMZ_INS_VGFMA,
	SYSTEMZ_INS_VGFMAB,
	SYSTEMZ_INS_VGFMAF,
	SYSTEMZ_INS_VGFMAG,
	SYSTEMZ_INS_VGFMAH,
	SYSTEMZ_INS_VGFMB,
	SYSTEMZ_INS_VGFMF,
	SYSTEMZ_INS_VGFMG,
	SYSTEMZ_INS_VGFMH,
	SYSTEMZ_INS_VGM,
	SYSTEMZ_INS_VGMB,
	SYSTEMZ_INS_VGMF,
	SYSTEMZ_INS_VGMG,
	SYSTEMZ_INS_VGMH,
	SYSTEMZ_INS_VISTR,
	SYSTEMZ_INS_VISTRB,
	SYSTEMZ_INS_VISTRBS,
	SYSTEMZ_INS_VISTRF,
	SYSTEMZ_INS_VISTRFS,
	SYSTEMZ_INS_VISTRH,
	SYSTEMZ_INS_VISTRHS,
	SYSTEMZ_INS_VL,
	SYSTEMZ_INS_VLBB,
	SYSTEMZ_INS_VLBR,
	SYSTEMZ_INS_VLBRF,
	SYSTEMZ_INS_VLBRG,
	SYSTEMZ_INS_VLBRH,
	SYSTEMZ_INS_VLBRQ,
	SYSTEMZ_INS_VLBRREP,
	SYSTEMZ_INS_VLBRREPF,
	SYSTEMZ_INS_VLBRREPG,
	SYSTEMZ_INS_VLBRREPH,
	SYSTEMZ_INS_VLC,
	SYSTEMZ_INS_VLCB,
	SYSTEMZ_INS_VLCF,
	SYSTEMZ_INS_VLCG,
	SYSTEMZ_INS_VLCH,
	SYSTEMZ_INS_VLDE,
	SYSTEMZ_INS_VLDEB,
	SYSTEMZ_INS_VLEB,
	SYSTEMZ_INS_VLEBRF,
	SYSTEMZ_INS_VLEBRG,
	SYSTEMZ_INS_VLEBRH,
	SYSTEMZ_INS_VLED,
	SYSTEMZ_INS_VLEDB,
	SYSTEMZ_INS_VLEF,
	SYSTEMZ_INS_VLEG,
	SYSTEMZ_INS_VLEH,
	SYSTEMZ_INS_VLEIB,
	SYSTEMZ_INS_VLEIF,
	SYSTEMZ_INS_VLEIG,
	SYSTEMZ_INS_VLEIH,
	SYSTEMZ_INS_VLER,
	SYSTEMZ_INS_VLERF,
	SYSTEMZ_INS_VLERG,
	SYSTEMZ_INS_VLERH,
	SYSTEMZ_INS_VLGV,
	SYSTEMZ_INS_VLGVB,
	SYSTEMZ_INS_VLGVF,
	SYSTEMZ_INS_VLGVG,
	SYSTEMZ_INS_VLGVH,
	SYSTEMZ_INS_VLIP,
	SYSTEMZ_INS_VLL,
	SYSTEMZ_INS_VLLEBRZ,
	SYSTEMZ_INS_VLLEBRZE,
	SYSTEMZ_INS_VLLEBRZF,
	SYSTEMZ_INS_VLLEBRZG,
	SYSTEMZ_INS_VLLEBRZH,
	SYSTEMZ_INS_VLLEZ,
	SYSTEMZ_INS_VLLEZB,
	SYSTEMZ_INS_VLLEZF,
	SYSTEMZ_INS_VLLEZG,
	SYSTEMZ_INS_VLLEZH,
	SYSTEMZ_INS_VLLEZLF,
	SYSTEMZ_INS_VLM,
	SYSTEMZ_INS_VLP,
	SYSTEMZ_INS_VLPB,
	SYSTEMZ_INS_VLPF,
	SYSTEMZ_INS_VLPG,
	SYSTEMZ_INS_VLPH,
	SYSTEMZ_INS_VLR,
	SYSTEMZ_INS_VLREP,
	SYSTEMZ_INS_VLREPB,
	SYSTEMZ_INS_VLREPF,
	SYSTEMZ_INS_VLREPG,
	SYSTEMZ_INS_VLREPH,
	SYSTEMZ_INS_VLRL,
	SYSTEMZ_INS_VLRLR,
	SYSTEMZ_INS_VLVG,
	SYSTEMZ_INS_VLVGB,
	SYSTEMZ_INS_VLVGF,
	SYSTEMZ_INS_VLVGG,
	SYSTEMZ_INS_VLVGH,
	SYSTEMZ_INS_VLVGP,
	SYSTEMZ_INS_VMAE,
	SYSTEMZ_INS_VMAEB,
	SYSTEMZ_INS_VMAEF,
	SYSTEMZ_INS_VMAEH,
	SYSTEMZ_INS_VMAH,
	SYSTEMZ_INS_VMAHB,
	SYSTEMZ_INS_VMAHF,
	SYSTEMZ_INS_VMAHH,
	SYSTEMZ_INS_VMAL,
	SYSTEMZ_INS_VMALB,
	SYSTEMZ_INS_VMALE,
	SYSTEMZ_INS_VMALEB,
	SYSTEMZ_INS_VMALEF,
	SYSTEMZ_INS_VMALEH,
	SYSTEMZ_INS_VMALF,
	SYSTEMZ_INS_VMALH,
	SYSTEMZ_INS_VMALHB,
	SYSTEMZ_INS_VMALHF,
	SYSTEMZ_INS_VMALHH,
	SYSTEMZ_INS_VMALHW,
	SYSTEMZ_INS_VMALO,
	SYSTEMZ_INS_VMALOB,
	SYSTEMZ_INS_VMALOF,
	SYSTEMZ_INS_VMALOH,
	SYSTEMZ_INS_VMAO,
	SYSTEMZ_INS_VMAOB,
	SYSTEMZ_INS_VMAOF,
	SYSTEMZ_INS_VMAOH,
	SYSTEMZ_INS_VME,
	SYSTEMZ_INS_VMEB,
	SYSTEMZ_INS_VMEF,
	SYSTEMZ_INS_VMEH,
	SYSTEMZ_INS_VMH,
	SYSTEMZ_INS_VMHB,
	SYSTEMZ_INS_VMHF,
	SYSTEMZ_INS_VMHH,
	SYSTEMZ_INS_VML,
	SYSTEMZ_INS_VMLB,
	SYSTEMZ_INS_VMLE,
	SYSTEMZ_INS_VMLEB,
	SYSTEMZ_INS_VMLEF,
	SYSTEMZ_INS_VMLEH,
	SYSTEMZ_INS_VMLF,
	SYSTEMZ_INS_VMLH,
	SYSTEMZ_INS_VMLHB,
	SYSTEMZ_INS_VMLHF,
	SYSTEMZ_INS_VMLHH,
	SYSTEMZ_INS_VMLHW,
	SYSTEMZ_INS_VMLO,
	SYSTEMZ_INS_VMLOB,
	SYSTEMZ_INS_VMLOF,
	SYSTEMZ_INS_VMLOH,
	SYSTEMZ_INS_VMN,
	SYSTEMZ_INS_VMNB,
	SYSTEMZ_INS_VMNF,
	SYSTEMZ_INS_VMNG,
	SYSTEMZ_INS_VMNH,
	SYSTEMZ_INS_VMNL,
	SYSTEMZ_INS_VMNLB,
	SYSTEMZ_INS_VMNLF,
	SYSTEMZ_INS_VMNLG,
	SYSTEMZ_INS_VMNLH,
	SYSTEMZ_INS_VMO,
	SYSTEMZ_INS_VMOB,
	SYSTEMZ_INS_VMOF,
	SYSTEMZ_INS_VMOH,
	SYSTEMZ_INS_VMP,
	SYSTEMZ_INS_VMRH,
	SYSTEMZ_INS_VMRHB,
	SYSTEMZ_INS_VMRHF,
	SYSTEMZ_INS_VMRHG,
	SYSTEMZ_INS_VMRHH,
	SYSTEMZ_INS_VMRL,
	SYSTEMZ_INS_VMRLB,
	SYSTEMZ_INS_VMRLF,
	SYSTEMZ_INS_VMRLG,
	SYSTEMZ_INS_VMRLH,
	SYSTEMZ_INS_VMSL,
	SYSTEMZ_INS_VMSLG,
	SYSTEMZ_INS_VMSP,
	SYSTEMZ_INS_VMX,
	SYSTEMZ_INS_VMXB,
	SYSTEMZ_INS_VMXF,
	SYSTEMZ_INS_VMXG,
	SYSTEMZ_INS_VMXH,
	SYSTEMZ_INS_VMXL,
	SYSTEMZ_INS_VMXLB,
	SYSTEMZ_INS_VMXLF,
	SYSTEMZ_INS_VMXLG,
	SYSTEMZ_INS_VMXLH,
	SYSTEMZ_INS_VN,
	SYSTEMZ_INS_VNC,
	SYSTEMZ_INS_VNN,
	SYSTEMZ_INS_VNO,
	SYSTEMZ_INS_VNX,
	SYSTEMZ_INS_VO,
	SYSTEMZ_INS_VOC,
	SYSTEMZ_INS_VONE,
	SYSTEMZ_INS_VPDI,
	SYSTEMZ_INS_VPERM,
	SYSTEMZ_INS_VPK,
	SYSTEMZ_INS_VPKF,
	SYSTEMZ_INS_VPKG,
	SYSTEMZ_INS_VPKH,
	SYSTEMZ_INS_VPKLS,
	SYSTEMZ_INS_VPKLSF,
	SYSTEMZ_INS_VPKLSFS,
	SYSTEMZ_INS_VPKLSG,
	SYSTEMZ_INS_VPKLSGS,
	SYSTEMZ_INS_VPKLSH,
	SYSTEMZ_INS_VPKLSHS,
	SYSTEMZ_INS_VPKS,
	SYSTEMZ_INS_VPKSF,
	SYSTEMZ_INS_VPKSFS,
	SYSTEMZ_INS_VPKSG,
	SYSTEMZ_INS_VPKSGS,
	SYSTEMZ_INS_VPKSH,
	SYSTEMZ_INS_VPKSHS,
	SYSTEMZ_INS_VPKZ,
	SYSTEMZ_INS_VPKZR,
	SYSTEMZ_INS_VPOPCT,
	SYSTEMZ_INS_VPOPCTB,
	SYSTEMZ_INS_VPOPCTF,
	SYSTEMZ_INS_VPOPCTG,
	SYSTEMZ_INS_VPOPCTH,
	SYSTEMZ_INS_VPSOP,
	SYSTEMZ_INS_VREP,
	SYSTEMZ_INS_VREPB,
	SYSTEMZ_INS_VREPF,
	SYSTEMZ_INS_VREPG,
	SYSTEMZ_INS_VREPH,
	SYSTEMZ_INS_VREPI,
	SYSTEMZ_INS_VREPIB,
	SYSTEMZ_INS_VREPIF,
	SYSTEMZ_INS_VREPIG,
	SYSTEMZ_INS_VREPIH,
	SYSTEMZ_INS_VRP,
	SYSTEMZ_INS_VS,
	SYSTEMZ_INS_VSB,
	SYSTEMZ_INS_VSBCBI,
	SYSTEMZ_INS_VSBCBIQ,
	SYSTEMZ_INS_VSBI,
	SYSTEMZ_INS_VSBIQ,
	SYSTEMZ_INS_VSCBI,
	SYSTEMZ_INS_VSCBIB,
	SYSTEMZ_INS_VSCBIF,
	SYSTEMZ_INS_VSCBIG,
	SYSTEMZ_INS_VSCBIH,
	SYSTEMZ_INS_VSCBIQ,
	SYSTEMZ_INS_VSCEF,
	SYSTEMZ_INS_VSCEG,
	SYSTEMZ_INS_VSCHDP,
	SYSTEMZ_INS_VSCHP,
	SYSTEMZ_INS_VSCHSP,
	SYSTEMZ_INS_VSCHXP,
	SYSTEMZ_INS_VSCSHP,
	SYSTEMZ_INS_VSDP,
	SYSTEMZ_INS_VSEG,
	SYSTEMZ_INS_VSEGB,
	SYSTEMZ_INS_VSEGF,
	SYSTEMZ_INS_VSEGH,
	SYSTEMZ_INS_VSEL,
	SYSTEMZ_INS_VSF,
	SYSTEMZ_INS_VSG,
	SYSTEMZ_INS_VSH,
	SYSTEMZ_INS_VSL,
	SYSTEMZ_INS_VSLB,
	SYSTEMZ_INS_VSLD,
	SYSTEMZ_INS_VSLDB,
	SYSTEMZ_INS_VSP,
	SYSTEMZ_INS_VSQ,
	SYSTEMZ_INS_VSRA,
	SYSTEMZ_INS_VSRAB,
	SYSTEMZ_INS_VSRD,
	SYSTEMZ_INS_VSRL,
	SYSTEMZ_INS_VSRLB,
	SYSTEMZ_INS_VSRP,
	SYSTEMZ_INS_VSRPR,
	SYSTEMZ_INS_VST,
	SYSTEMZ_INS_VSTBR,
	SYSTEMZ_INS_VSTBRF,
	SYSTEMZ_INS_VSTBRG,
	SYSTEMZ_INS_VSTBRH,
	SYSTEMZ_INS_VSTBRQ,
	SYSTEMZ_INS_VSTEB,
	SYSTEMZ_INS_VSTEBRF,
	SYSTEMZ_INS_VSTEBRG,
	SYSTEMZ_INS_VSTEBRH,
	SYSTEMZ_INS_VSTEF,
	SYSTEMZ_INS_VSTEG,
	SYSTEMZ_INS_VSTEH,
	SYSTEMZ_INS_VSTER,
	SYSTEMZ_INS_VSTERF,
	SYSTEMZ_INS_VSTERG,
	SYSTEMZ_INS_VSTERH,
	SYSTEMZ_INS_VSTL,
	SYSTEMZ_INS_VSTM,
	SYSTEMZ_INS_VSTRC,
	SYSTEMZ_INS_VSTRCB,
	SYSTEMZ_INS_VSTRCBS,
	SYSTEMZ_INS_VSTRCF,
	SYSTEMZ_INS_VSTRCFS,
	SYSTEMZ_INS_VSTRCH,
	SYSTEMZ_INS_VSTRCHS,
	SYSTEMZ_INS_VSTRCZB,
	SYSTEMZ_INS_VSTRCZBS,
	SYSTEMZ_INS_VSTRCZF,
	SYSTEMZ_INS_VSTRCZFS,
	SYSTEMZ_INS_VSTRCZH,
	SYSTEMZ_INS_VSTRCZHS,
	SYSTEMZ_INS_VSTRL,
	SYSTEMZ_INS_VSTRLR,
	SYSTEMZ_INS_VSTRS,
	SYSTEMZ_INS_VSTRSB,
	SYSTEMZ_INS_VSTRSF,
	SYSTEMZ_INS_VSTRSH,
	SYSTEMZ_INS_VSTRSZB,
	SYSTEMZ_INS_VSTRSZF,
	SYSTEMZ_INS_VSTRSZH,
	SYSTEMZ_INS_VSUM,
	SYSTEMZ_INS_VSUMB,
	SYSTEMZ_INS_VSUMG,
	SYSTEMZ_INS_VSUMGF,
	SYSTEMZ_INS_VSUMGH,
	SYSTEMZ_INS_VSUMH,
	SYSTEMZ_INS_VSUMQ,
	SYSTEMZ_INS_VSUMQF,
	SYSTEMZ_INS_VSUMQG,
	SYSTEMZ_INS_VTM,
	SYSTEMZ_INS_VTP,
	SYSTEMZ_INS_VUPH,
	SYSTEMZ_INS_VUPHB,
	SYSTEMZ_INS_VUPHF,
	SYSTEMZ_INS_VUPHH,
	SYSTEMZ_INS_VUPKZ,
	SYSTEMZ_INS_VUPKZH,
	SYSTEMZ_INS_VUPKZL,
	SYSTEMZ_INS_VUPL,
	SYSTEMZ_INS_VUPLB,
	SYSTEMZ_INS_VUPLF,
	SYSTEMZ_INS_VUPLH,
	SYSTEMZ_INS_VUPLHB,
	SYSTEMZ_INS_VUPLHF,
	SYSTEMZ_INS_VUPLHH,
	SYSTEMZ_INS_VUPLHW,
	SYSTEMZ_INS_VUPLL,
	SYSTEMZ_INS_VUPLLB,
	SYSTEMZ_INS_VUPLLF,
	SYSTEMZ_INS_VUPLLH,
	SYSTEMZ_INS_VX,
	SYSTEMZ_INS_VZERO,
	SYSTEMZ_INS_WCDGB,
	SYSTEMZ_INS_WCDLGB,
	SYSTEMZ_INS_WCEFB,
	SYSTEMZ_INS_WCELFB,
	SYSTEMZ_INS_WCFEB,
	SYSTEMZ_INS_WCGDB,
	SYSTEMZ_INS_WCLFEB,
	SYSTEMZ_INS_WCLGDB,
	SYSTEMZ_INS_WFADB,
	SYSTEMZ_INS_WFASB,
	SYSTEMZ_INS_WFAXB,
	SYSTEMZ_INS_WFC,
	SYSTEMZ_INS_WFCDB,
	SYSTEMZ_INS_WFCEDB,
	SYSTEMZ_INS_WFCEDBS,
	SYSTEMZ_INS_WFCESB,
	SYSTEMZ_INS_WFCESBS,
	SYSTEMZ_INS_WFCEXB,
	SYSTEMZ_INS_WFCEXBS,
	SYSTEMZ_INS_WFCHDB,
	SYSTEMZ_INS_WFCHDBS,
	SYSTEMZ_INS_WFCHEDB,
	SYSTEMZ_INS_WFCHEDBS,
	SYSTEMZ_INS_WFCHESB,
	SYSTEMZ_INS_WFCHESBS,
	SYSTEMZ_INS_WFCHEXB,
	SYSTEMZ_INS_WFCHEXBS,
	SYSTEMZ_INS_WFCHSB,
	SYSTEMZ_INS_WFCHSBS,
	SYSTEMZ_INS_WFCHXB,
	SYSTEMZ_INS_WFCHXBS,
	SYSTEMZ_INS_WFCSB,
	SYSTEMZ_INS_WFCXB,
	SYSTEMZ_INS_WFDDB,
	SYSTEMZ_INS_WFDSB,
	SYSTEMZ_INS_WFDXB,
	SYSTEMZ_INS_WFIDB,
	SYSTEMZ_INS_WFISB,
	SYSTEMZ_INS_WFIXB,
	SYSTEMZ_INS_WFK,
	SYSTEMZ_INS_WFKDB,
	SYSTEMZ_INS_WFKEDB,
	SYSTEMZ_INS_WFKEDBS,
	SYSTEMZ_INS_WFKESB,
	SYSTEMZ_INS_WFKESBS,
	SYSTEMZ_INS_WFKEXB,
	SYSTEMZ_INS_WFKEXBS,
	SYSTEMZ_INS_WFKHDB,
	SYSTEMZ_INS_WFKHDBS,
	SYSTEMZ_INS_WFKHEDB,
	SYSTEMZ_INS_WFKHEDBS,
	SYSTEMZ_INS_WFKHESB,
	SYSTEMZ_INS_WFKHESBS,
	SYSTEMZ_INS_WFKHEXB,
	SYSTEMZ_INS_WFKHEXBS,
	SYSTEMZ_INS_WFKHSB,
	SYSTEMZ_INS_WFKHSBS,
	SYSTEMZ_INS_WFKHXB,
	SYSTEMZ_INS_WFKHXBS,
	SYSTEMZ_INS_WFKSB,
	SYSTEMZ_INS_WFKXB,
	SYSTEMZ_INS_WFLCDB,
	SYSTEMZ_INS_WFLCSB,
	SYSTEMZ_INS_WFLCXB,
	SYSTEMZ_INS_WFLLD,
	SYSTEMZ_INS_WFLLS,
	SYSTEMZ_INS_WFLNDB,
	SYSTEMZ_INS_WFLNSB,
	SYSTEMZ_INS_WFLNXB,
	SYSTEMZ_INS_WFLPDB,
	SYSTEMZ_INS_WFLPSB,
	SYSTEMZ_INS_WFLPXB,
	SYSTEMZ_INS_WFLRD,
	SYSTEMZ_INS_WFLRX,
	SYSTEMZ_INS_WFMADB,
	SYSTEMZ_INS_WFMASB,
	SYSTEMZ_INS_WFMAXB,
	SYSTEMZ_INS_WFMAXDB,
	SYSTEMZ_INS_WFMAXSB,
	SYSTEMZ_INS_WFMAXXB,
	SYSTEMZ_INS_WFMDB,
	SYSTEMZ_INS_WFMINDB,
	SYSTEMZ_INS_WFMINSB,
	SYSTEMZ_INS_WFMINXB,
	SYSTEMZ_INS_WFMSB,
	SYSTEMZ_INS_WFMSDB,
	SYSTEMZ_INS_WFMSSB,
	SYSTEMZ_INS_WFMSXB,
	SYSTEMZ_INS_WFMXB,
	SYSTEMZ_INS_WFNMADB,
	SYSTEMZ_INS_WFNMASB,
	SYSTEMZ_INS_WFNMAXB,
	SYSTEMZ_INS_WFNMSDB,
	SYSTEMZ_INS_WFNMSSB,
	SYSTEMZ_INS_WFNMSXB,
	SYSTEMZ_INS_WFPSODB,
	SYSTEMZ_INS_WFPSOSB,
	SYSTEMZ_INS_WFPSOXB,
	SYSTEMZ_INS_WFSDB,
	SYSTEMZ_INS_WFSQDB,
	SYSTEMZ_INS_WFSQSB,
	SYSTEMZ_INS_WFSQXB,
	SYSTEMZ_INS_WFSSB,
	SYSTEMZ_INS_WFSXB,
	SYSTEMZ_INS_WFTCIDB,
	SYSTEMZ_INS_WFTCISB,
	SYSTEMZ_INS_WFTCIXB,
	SYSTEMZ_INS_WLDEB,
	SYSTEMZ_INS_WLEDB,
	SYSTEMZ_INS_X,
	SYSTEMZ_INS_XC,
	SYSTEMZ_INS_XG,
	SYSTEMZ_INS_XGR,
	SYSTEMZ_INS_XGRK,
	SYSTEMZ_INS_XI,
	SYSTEMZ_INS_XIHF,
	SYSTEMZ_INS_XILF,
	SYSTEMZ_INS_XIY,
	SYSTEMZ_INS_XR,
	SYSTEMZ_INS_XRK,
	SYSTEMZ_INS_XSCH,
	SYSTEMZ_INS_XY,
	SYSTEMZ_INS_ZAP,

	// clang-format on
	// generated content <SystemZGenCSInsnEnum.inc> end

	SYSTEMZ_INS_ENDING,

	SYSTEMZ_INS_ALIAS_BEGIN,
	// generated content <SystemZGenCSAliasEnum.inc> begin
	// clang-format off

	SYSTEMZ_INS_ALIAS_VISTRB, // Real instr.: SYSTEMZ_VISTRB
	SYSTEMZ_INS_ALIAS_VISTR, // Real instr.: SYSTEMZ_VISTR
	SYSTEMZ_INS_ALIAS_VFEEB, // Real instr.: SYSTEMZ_VFEEB
	SYSTEMZ_INS_ALIAS_VFEE, // Real instr.: SYSTEMZ_VFEE
	SYSTEMZ_INS_ALIAS_VFAEB, // Real instr.: SYSTEMZ_VFAEB
	SYSTEMZ_INS_ALIAS_VFAEBS, // Real instr.: SYSTEMZ_VFAEBS
	SYSTEMZ_INS_ALIAS_VFAE, // Real instr.: SYSTEMZ_VFAE
	SYSTEMZ_INS_ALIAS_VSTRSB, // Real instr.: SYSTEMZ_VSTRSB
	SYSTEMZ_INS_ALIAS_VSTRS, // Real instr.: SYSTEMZ_VSTRS
	SYSTEMZ_INS_ALIAS_VSTRCB, // Real instr.: SYSTEMZ_VSTRCB
	SYSTEMZ_INS_ALIAS_VSTRCBS, // Real instr.: SYSTEMZ_VSTRCBS
	SYSTEMZ_INS_ALIAS_VSTRC, // Real instr.: SYSTEMZ_VSTRC
	SYSTEMZ_INS_ALIAS_VFAEH, // Real instr.: SYSTEMZ_VFAEH
	SYSTEMZ_INS_ALIAS_VFAEHS, // Real instr.: SYSTEMZ_VFAEHS
	SYSTEMZ_INS_ALIAS_VFAEF, // Real instr.: SYSTEMZ_VFAEF
	SYSTEMZ_INS_ALIAS_VFAEFS, // Real instr.: SYSTEMZ_VFAEFS
	SYSTEMZ_INS_ALIAS_VFAEZB, // Real instr.: SYSTEMZ_VFAEZB
	SYSTEMZ_INS_ALIAS_VFAEZBS, // Real instr.: SYSTEMZ_VFAEZBS
	SYSTEMZ_INS_ALIAS_VFAEZH, // Real instr.: SYSTEMZ_VFAEZH
	SYSTEMZ_INS_ALIAS_VFAEZHS, // Real instr.: SYSTEMZ_VFAEZHS
	SYSTEMZ_INS_ALIAS_VFAEZF, // Real instr.: SYSTEMZ_VFAEZF
	SYSTEMZ_INS_ALIAS_VFAEZFS, // Real instr.: SYSTEMZ_VFAEZFS
	SYSTEMZ_INS_ALIAS_VFEEH, // Real instr.: SYSTEMZ_VFEEH
	SYSTEMZ_INS_ALIAS_VFEEF, // Real instr.: SYSTEMZ_VFEEF
	SYSTEMZ_INS_ALIAS_VFENE, // Real instr.: SYSTEMZ_VFENE
	SYSTEMZ_INS_ALIAS_VFENEB, // Real instr.: SYSTEMZ_VFENEB
	SYSTEMZ_INS_ALIAS_VFENEH, // Real instr.: SYSTEMZ_VFENEH
	SYSTEMZ_INS_ALIAS_VFENEF, // Real instr.: SYSTEMZ_VFENEF
	SYSTEMZ_INS_ALIAS_VISTRH, // Real instr.: SYSTEMZ_VISTRH
	SYSTEMZ_INS_ALIAS_VISTRF, // Real instr.: SYSTEMZ_VISTRF
	SYSTEMZ_INS_ALIAS_VSTRCH, // Real instr.: SYSTEMZ_VSTRCH
	SYSTEMZ_INS_ALIAS_VSTRCHS, // Real instr.: SYSTEMZ_VSTRCHS
	SYSTEMZ_INS_ALIAS_VSTRCF, // Real instr.: SYSTEMZ_VSTRCF
	SYSTEMZ_INS_ALIAS_VSTRCFS, // Real instr.: SYSTEMZ_VSTRCFS
	SYSTEMZ_INS_ALIAS_VSTRCZB, // Real instr.: SYSTEMZ_VSTRCZB
	SYSTEMZ_INS_ALIAS_VSTRCZBS, // Real instr.: SYSTEMZ_VSTRCZBS
	SYSTEMZ_INS_ALIAS_VSTRCZH, // Real instr.: SYSTEMZ_VSTRCZH
	SYSTEMZ_INS_ALIAS_VSTRCZHS, // Real instr.: SYSTEMZ_VSTRCZHS
	SYSTEMZ_INS_ALIAS_VSTRCZF, // Real instr.: SYSTEMZ_VSTRCZF
	SYSTEMZ_INS_ALIAS_VSTRCZFS, // Real instr.: SYSTEMZ_VSTRCZFS
	SYSTEMZ_INS_ALIAS_VSTRSH, // Real instr.: SYSTEMZ_VSTRSH
	SYSTEMZ_INS_ALIAS_VSTRSF, // Real instr.: SYSTEMZ_VSTRSF

	// clang-format on
	// generated content <SystemZGenCSAliasEnum.inc> end

	// Hard-coded alias come here

	SYSTEMZ_INS_ALIAS_END,
} systemz_insn;

/// Group of SystemZ instructions
typedef enum systemz_insn_group {
	SYSTEMZ_GRP_INVALID = 0, ///< = CS_GRP_INVALID

	// Generic groups
	// all jump instructions (conditional+direct+indirect jumps)
	SYSTEMZ_GRP_JUMP, ///< = CS_GRP_JUMP
	SYSTEMZ_GRP_CALL, ///< CS_GRP_CALL
	SYSTEMZ_GRP_RET, ///< CS_GRP_RET
	SYSTEMZ_GRP_INT, ///< CS_GRP_INT
	SYSTEMZ_GRP_IRET, ///< CS_GRP_IRET
	SYSTEMZ_GRP_PRIVILEGE, ///< CS_GRP_PRIVILEGE
	SYSTEMZ_GRP_BRANCH_RELATIVE, ///< CS_GRP_BRANCH_RELATIVE
	// generated content <SystemZGenCSFeatureEnum.inc> begin
	// clang-format off

	SYSTEMZ_FEATURE_FEATURESOFTFLOAT = 128,
	SYSTEMZ_FEATURE_FEATUREBACKCHAIN,
	SYSTEMZ_FEATURE_FEATUREDISTINCTOPS,
	SYSTEMZ_FEATURE_FEATUREFASTSERIALIZATION,
	SYSTEMZ_FEATURE_FEATUREFPEXTENSION,
	SYSTEMZ_FEATURE_FEATUREHIGHWORD,
	SYSTEMZ_FEATURE_FEATUREINTERLOCKEDACCESS1,
	SYSTEMZ_FEATURE_FEATURELOADSTOREONCOND,
	SYSTEMZ_FEATURE_FEATUREPOPULATIONCOUNT,
	SYSTEMZ_FEATURE_FEATUREMESSAGESECURITYASSIST3,
	SYSTEMZ_FEATURE_FEATUREMESSAGESECURITYASSIST4,
	SYSTEMZ_FEATURE_FEATURERESETREFERENCEBITSMULTIPLE,
	SYSTEMZ_FEATURE_FEATUREEXECUTIONHINT,
	SYSTEMZ_FEATURE_FEATURELOADANDTRAP,
	SYSTEMZ_FEATURE_FEATUREMISCELLANEOUSEXTENSIONS,
	SYSTEMZ_FEATURE_FEATUREPROCESSORASSIST,
	SYSTEMZ_FEATURE_FEATURETRANSACTIONALEXECUTION,
	SYSTEMZ_FEATURE_FEATUREDFPZONEDCONVERSION,
	SYSTEMZ_FEATURE_FEATUREENHANCEDDAT2,
	SYSTEMZ_FEATURE_FEATURELOADANDZERORIGHTMOSTBYTE,
	SYSTEMZ_FEATURE_FEATURELOADSTOREONCOND2,
	SYSTEMZ_FEATURE_FEATUREMESSAGESECURITYASSIST5,
	SYSTEMZ_FEATURE_FEATUREDFPPACKEDCONVERSION,
	SYSTEMZ_FEATURE_FEATUREVECTOR,
	SYSTEMZ_FEATURE_FEATUREMISCELLANEOUSEXTENSIONS2,
	SYSTEMZ_FEATURE_FEATUREGUARDEDSTORAGE,
	SYSTEMZ_FEATURE_FEATUREMESSAGESECURITYASSIST7,
	SYSTEMZ_FEATURE_FEATUREMESSAGESECURITYASSIST8,
	SYSTEMZ_FEATURE_FEATUREVECTORENHANCEMENTS1,
	SYSTEMZ_FEATURE_FEATUREVECTORPACKEDDECIMAL,
	SYSTEMZ_FEATURE_FEATUREINSERTREFERENCEBITSMULTIPLE,
	SYSTEMZ_FEATURE_FEATUREMISCELLANEOUSEXTENSIONS3,
	SYSTEMZ_FEATURE_FEATUREMESSAGESECURITYASSIST9,
	SYSTEMZ_FEATURE_FEATUREVECTORENHANCEMENTS2,
	SYSTEMZ_FEATURE_FEATUREVECTORPACKEDDECIMALENHANCEMENT,
	SYSTEMZ_FEATURE_FEATUREENHANCEDSORT,
	SYSTEMZ_FEATURE_FEATUREDEFLATECONVERSION,
	SYSTEMZ_FEATURE_FEATUREVECTORPACKEDDECIMALENHANCEMENT2,
	SYSTEMZ_FEATURE_FEATURENNPASSIST,
	SYSTEMZ_FEATURE_FEATUREBEARENHANCEMENT,
	SYSTEMZ_FEATURE_FEATURERESETDATPROTECTION,
	SYSTEMZ_FEATURE_FEATUREPROCESSORACTIVITYINSTRUMENTATION,

	// clang-format on
	// generated content <SystemZGenCSFeatureEnum.inc> end

	SYSTEMZ_GRP_ENDING, // <-- mark the end of the list of groups
} systemz_insn_group;

#ifdef CAPSTONE_SYSTEMZ_COMPAT_HEADER
#include "systemz_compatibility.h"
#endif

#ifdef __cplusplus
}
#endif

#endif
