#ifndef PRE_LEXER_H
#define PRE_LEXER_H

#include <glib.h>

/* Structure to store lexer state. */
typedef struct
{
    gchar* stream;		/* Pointer to the local copy of input string. */
    guint length;		/* Length of input string; stored to reduce calls to strlen(). */
    guint next_index;		/* Index of next (to be read) character from input. */
    guint mark_index;		/* Location, last marked. Useful for getting substrings as part of highlighting */
} PreLexerState;

/* Enum for tokens generated by pre-lexer and lexer. */
typedef enum
{
    T_UNKNOWN=0,		//Unknown

    /* These are all Pre-Lexer tokens, returned by pre-lexer */
    PL_DECIMAL,			//Decimal saperator
    PL_DIGIT,			//Decimal digit
    PL_HEX,			//A-F of Hex digits
    PL_SUPER_DIGIT,		//Super digits
    PL_SUPER_MINUS,		//Super minus
    PL_SUB_DIGIT,		//Sub digits
    PL_FRACTION,		//Fractions
    PL_DEGREE,			//Degree
    PL_MINUTE,			//Minutes
    PL_SECOND,		//10	//Seconds
    PL_LETTER,			//Alphabets
    PL_EOS,			//End of stream. Yay!!
    PL_SKIP,			//Skip this symbol (whitespace or newline).

    /* These are all tokens, returned by Lexer. */
    T_ADD,			//Plus
    T_SUBTRACT,			//Minus
    T_MULTIPLY,			//Multiply
    T_DIV,			//Divide (note can't use T_DIVIDE as it is defined in *BSD as an "integer divide fault" trap type value)
    T_MOD,			//Modulus
    T_L_FLOOR,			//Floor ( Left )
    T_R_FLOOR,		//20	//Floor ( Right )
    T_L_CEILING,		//Ceiling ( Left )
    T_R_CEILING,		//Ceiling ( Right )
    T_ROOT,			//Square root
    T_ROOT_3,			//Cube root
    T_ROOT_4,			//Fourth root
    T_NOT,			//Bitwise NOT
    T_AND,			//Bitwise AND
    T_OR,			//Bitwise OR
    T_XOR,			//Bitwise XOR
    T_IN,		//30	//IN ( for converter )
    T_NUMBER,			//Number
    T_SUP_NUMBER,		//Super Number
    T_NSUP_NUMBER,		//Negative Super Number
    T_SUB_NUMBER,		//Sub Number
    T_FUNCTION,			//Function
    T_VARIABLE,			//Variable name
    T_ASSIGN,			//=
    T_L_R_BRACKET,	//40	//(
    T_R_R_BRACKET,		//)
    T_L_S_BRACKET,		//[
    T_R_S_BRACKET,		//]
    T_L_C_BRACKET,		//{
    T_R_C_BRACKET,		//}
    T_ABS,			//|
    T_POWER,			//^
    T_FACTORIAL,		//!
    T_PERCENTAGE	//49	//%
} LexerTokenType;

/* Creates a scanner state. Useful when multiple scanners are in action. */
PreLexerState* pl_create_scanner(const gchar*);

/* Destroy and free memory used by LexerState object. */
void pl_destroy_scanner(PreLexerState*);

/* Roll back last scanned unichar. */
void pl_roll_back(PreLexerState*);

/* Get validated gunichar from input stream. */
gunichar pl_get_next_gunichar(PreLexerState*);

/* Set marker index. To be used for highlighting and error reporting. */
void pl_set_marker(PreLexerState*);

/* Get marked substring. To be used for error reporting. */
gchar* pl_get_marked_substring(const PreLexerState*);

/* Get next Pre-Lexer token from stream. */
LexerTokenType pl_get_next_token(PreLexerState*);

#endif /* PRE_LEXER_H */
