/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt61pci
	Abstract: rt61pci device specific routines.
	Supported chipsets: RT2561, RT2561s, RT2661.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/firmware.h>
#include <linux/ethtool.h>
#include <linux/crc-itu-t.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

#include <asm/io.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME "rt61pci"

#ifdef CONFIG_RT61PCI_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT61PCI_DEBUG */

#ifdef CONFIG_RT61PCI_BUTTON
#define CONFIG_RT2X00_BUTTON
#endif /* CONFIG_RT61PCI_BUTTON */

#include "rt2x00.h"
#include "rt2x00pci.h"
#include "rt61pci.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers PHY_CSR3 and PHY_CSR4 to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 * The caller to these register access functions, should take precautions
 * for the correct byte ordering of the values.
 */
static inline void rt2x00_register_read(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 *value)
{
	readl((void*)(rt2x00dev->csr_addr + MAC_CSR0));
	*value = le32_to_cpu(readl((void*)(rt2x00dev->csr_addr + offset)));
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 *value, const u16 length)
{
	readl((void*)(rt2x00dev->csr_addr + MAC_CSR0));
	memcpy_fromio(value, (void*)(rt2x00dev->csr_addr + offset), length);
}

static inline void rt2x00_register_write(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, const u32 value)
{
	readl((void*)(rt2x00dev->csr_addr + MAC_CSR0));
	writel(cpu_to_le32(value), (void*)(rt2x00dev->csr_addr + offset));
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_dev *rt2x00dev,
	const unsigned long offset, u32 *value, const u16 length)
{
	readl((void*)(rt2x00dev->csr_addr + MAC_CSR0));
	memcpy_toio((void*)(rt2x00dev->csr_addr + offset), value, length);
}

static void rt2x00_bbp_write(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, const u8 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR3, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR3_BUSY))
			goto bbp_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR3 register busy. Write failed.\n");
	return;

bbp_write:
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR3_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 0);

	rt2x00_register_write(rt2x00dev, PHY_CSR3, reg);
}

static void rt2x00_bbp_read(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, u8 *value)
{
	u32 reg;
	unsigned int i;

	/*
	 * First request the register we wish to read from.
	 */
	reg =0;
	rt2x00_set_field32(&reg, PHY_CSR3_REGNUM, reg_id);
	rt2x00_set_field32(&reg, PHY_CSR3_BUSY, 1);
	rt2x00_set_field32(&reg, PHY_CSR3_READ_CONTROL, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR3, reg);

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR3, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR3_BUSY)) {
			*value = rt2x00_get_field32(reg, PHY_CSR3_VALUE);
			return;
		}
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR3 register busy. Read failed.\n");
	*value = 0xff;
}

static void rt2x00_rf_write(const struct rt2x00_dev *rt2x00dev,
	const u32 value)
{
	u32 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR4, &reg);
		if (!rt2x00_get_field32(reg, PHY_CSR4_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("RFCSR register busy. Write failed.\n");
	return;

rf_write:
	reg = 0;
	rt2x00_set_field32(&reg, PHY_CSR4_VALUE, value);
	rt2x00_set_field32(&reg, PHY_CSR4_NUMBER_OF_BITS, 21);
	rt2x00_set_field32(&reg, PHY_CSR4_IF_SELECT, 0);
	rt2x00_set_field32(&reg, PHY_CSR4_BUSY, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR4, reg);
}

static void rt2x00_mcu_request(const struct rt2x00_dev *rt2x00dev,
	const u8 command, const u8 token, const u8 arg0, const u8 arg1)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, H2M_MAILBOX_CSR, &reg);

	if (rt2x00_get_field32(reg, H2M_MAILBOX_CSR_OWNER)) {
		ERROR("mcu request error. Request 0x%02x failed for "
			"token 0x%02x.\n", command, token);
		return;
	}

	rt2x00_set_field32(&reg, H2M_MAILBOX_CSR_OWNER, 1);
	rt2x00_set_field32(&reg, H2M_MAILBOX_CSR_CMD_TOKEN, token);
	rt2x00_set_field32(&reg, H2M_MAILBOX_CSR_ARG0, arg0);
	rt2x00_set_field32(&reg, H2M_MAILBOX_CSR_ARG1, arg1);
	rt2x00_register_write(rt2x00dev, H2M_MAILBOX_CSR, reg);

	rt2x00_register_read(rt2x00dev, HOST_CMD_CSR, &reg);
	rt2x00_set_field32(&reg, HOST_CMD_CSR_HOST_COMMAND, command);
	rt2x00_set_field32(&reg, HOST_CMD_CSR_INTERRUPT_MCU, 1);
	rt2x00_register_write(rt2x00dev, HOST_CMD_CSR, reg);
}

/*
 * EEPROM access.
 * The EEPROM is being accessed by word index.
 * rt2x00_eeprom_read is the main access function,
 * which should be called from the rest of the module.
 * It will take the index number of the eeprom word
 * and the value in which the data should be stored.
 * This function does assume the bus_width attribute
 * within rt2x00_dev has been correctly set.
 */
static inline void rt2x00_eeprom_pulse_high(const struct rt2x00_dev *rt2x00dev,
	u32 *flags)
{
	rt2x00_set_field32(flags, E2PROM_CSR_DATA_CLOCK, 1);
	rt2x00_register_write(rt2x00dev, E2PROM_CSR, *flags);
	udelay(1);
}

static inline void rt2x00_eeprom_pulse_low(const struct rt2x00_dev *rt2x00dev,
	u32 *flags)
{
	rt2x00_set_field32(flags, E2PROM_CSR_DATA_CLOCK, 0);
	rt2x00_register_write(rt2x00dev, E2PROM_CSR, *flags);
	udelay(1);
}

static void rt2x00_eeprom_shift_out_bits(const struct rt2x00_dev *rt2x00dev,
	const u16 data, const u16 count)
{
	u32 flags;
	u32 mask =  1 << (count - 1);

	rt2x00_register_read(rt2x00dev, E2PROM_CSR, &flags);

	/*
	 * Clear data flags.
	 */
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_OUT, 0);

	/*
	 * Start writing all bits.
	 */
	do {
		/*
		 * Set the data_in bit only when required.
		 */
		if (data & mask)
			rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 1);
		else
			rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);

		rt2x00_register_write(rt2x00dev, E2PROM_CSR, flags);

		rt2x00_eeprom_pulse_high(rt2x00dev, &flags);
		rt2x00_eeprom_pulse_low(rt2x00dev, &flags);

		/*
		 * Shift to next bit.
		 */
		mask >>= 1;
	} while (mask);

	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);
	rt2x00_register_write(rt2x00dev, E2PROM_CSR, flags);
}

static void rt2x00_eeprom_shift_in_bits(const struct rt2x00_dev *rt2x00dev,
	u16 *data)
{
	u32 flags;
	unsigned int i;

	rt2x00_register_read(rt2x00dev, E2PROM_CSR, &flags);

	/*
	 * Clear data flags.
	 */
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_OUT, 0);

	/*
	 * Start reading all 16 bits.
	 */
	for (i = 0; i < 16; i++) {
		/*
		 * Shift to the next bit.
		 */
		*data <<= 1;

		rt2x00_eeprom_pulse_high(rt2x00dev, &flags);

		rt2x00_register_read(rt2x00dev, E2PROM_CSR, &flags);

		/*
		 * Clear data_in flag.
		 */
		rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);

		/*
		 * Set the data bit to 1 when the data_out flag is set.
		 */
		if (rt2x00_get_field32(flags, E2PROM_CSR_DATA_OUT))
			*data |= 1;

		rt2x00_eeprom_pulse_low(rt2x00dev, &flags);
	}
}

static void rt2x00_eeprom_read(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 *data)
{
	u32 flags;

	/*
	 * Clear all flags, and enable chip select.
	 */
	rt2x00_register_read(rt2x00dev, E2PROM_CSR, &flags);
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_OUT, 0);
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_CLOCK, 0);
	rt2x00_set_field32(&flags, E2PROM_CSR_CHIP_SELECT, 1);
	rt2x00_register_write(rt2x00dev, E2PROM_CSR, flags);

	/*
	 * kick a pulse.
	 */
	rt2x00_eeprom_pulse_high(rt2x00dev, &flags);
	rt2x00_eeprom_pulse_low(rt2x00dev, &flags);

	/*
	 * Select the read opcode and bus_width.
	 */
	rt2x00_eeprom_shift_out_bits(rt2x00dev, EEPROM_READ_OPCODE, 3);
	rt2x00_eeprom_shift_out_bits(rt2x00dev, word, rt2x00dev->eeprom_width);

	rt2x00_eeprom_shift_in_bits(rt2x00dev, data);

	/*
	 * Clear chip_select and data_in flags.
	 */
	rt2x00_register_read(rt2x00dev, E2PROM_CSR, &flags);
	rt2x00_set_field32(&flags, E2PROM_CSR_DATA_IN, 0);
	rt2x00_set_field32(&flags, E2PROM_CSR_CHIP_SELECT, 0);
	rt2x00_register_write(rt2x00dev, E2PROM_CSR, flags);

	/*
	 * kick a pulse.
	 */
	rt2x00_eeprom_pulse_high(rt2x00dev, &flags);
	rt2x00_eeprom_pulse_low(rt2x00dev, &flags);

	/*
	 * Byteorder.
	 */
	*data = le16_to_cpu(*data);
}

static void rt2x00_eeprom_multiread(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 *data, const u16 words)
{
	unsigned int i;

	for (i = 0; i < words; i++)
		rt2x00_eeprom_read(rt2x00dev, word + i, data++);
}

/*
 * TX/RX Descriptor access functions.
 */
static inline void* rt2x00pci_data_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline void* rt2x00pci_desc_addr(struct data_entry *entry)
{
	return entry->priv;
}

static inline void rt2x00_desc_read(const void *desc,
	const u8 word, u32 *value)
{
	*value = le32_to_cpu(((__le32*)desc)[word]);
}

static inline void rt2x00_desc_write(const void *desc,
	const u8 word, const u32 value)
{
	((__le32*)desc)[word] = cpu_to_le32(value);
}

/*
 * Ethtool handlers.
 */
static void rt61pci_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->fw_version, rt2x00_fw(&rt2x00dev->chip));
	strcpy(drvinfo->bus_info, pci_name(rt2x00dev_pci(rt2x00dev)));
	drvinfo->eedump_len = EEPROM_SIZE;
	drvinfo->regdump_len = CSR_REG_SIZE;
}

static int rt61pci_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt61pci_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_dev	*rt2x00dev = ieee80211_dev_hw_data(net_dev);

	rt2x00_register_multiread(rt2x00dev, CSR_REG_BASE, data, CSR_REG_SIZE);
	regs->version = rt2x00_rev(&rt2x00dev->chip);
	regs->len = CSR_REG_SIZE;
}

#ifdef CONFIG_RT61PCI_DEBUG
static u32 rt61pci_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt61pci_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT61PCI_DEBUG */

static int rt61pci_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt61pci_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	eeprom->magic = 0xbad;

	rt2x00_eeprom_multiread(rt2x00dev,
		eeprom->offset / sizeof(u16), (void*)data,
		eeprom->len / sizeof(u16));

	return 0;
}

static struct ethtool_ops rt61pci_ethtool_ops = {
	.get_drvinfo	= rt61pci_get_drvinfo,
	.get_regs_len	= rt61pci_get_regs_len,
	.get_regs	= rt61pci_get_regs,
#ifdef CONFIG_RT61PCI_DEBUG
	.get_msglevel	= rt61pci_get_msglevel,
	.set_msglevel	= rt61pci_set_msglevel,
#endif /* CONFIG_RT61PCI_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt61pci_get_eeprom_len,
	.get_eeprom	= rt61pci_get_eeprom,
	.get_perm_addr	= ethtool_op_get_perm_addr,
};

#ifdef CONFIG_RT61PCI_BUTTON
static int rt61pci_button_poll(unsigned long data)
{
	struct rt2x00_dev *rt2x00dev = (struct rt2x00_dev*)data;
	u32 reg;

	rt2x00_register_read(rt2x00dev, MAC_CSR13, &reg);
	return rt2x00_get_field32(reg, MAC_CSR13_BIT5);
}

static void rt61pci_button_enable_radio(unsigned long data)
{
	struct rt2x00_dev* rt2x00dev = (struct rt2x00_dev*)data;

	if (unlikely(!rt2x00dev))
		return;

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);

	/*
	 * Only continue if we have an active interface,
	 * either monitor or non-monitor should be present.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ||
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		NOTICE("Hardware button pressed, enabling radio.\n");
		rt61pci_enable_radio(rt2x00dev);
	}
}

static void rt61pci_button_disable_radio(unsigned long data)
{
	struct rt2x00_dev* rt2x00dev = (struct rt2x00_dev*)data;

	if (unlikely(!rt2x00dev))
		return;

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);

	/*
	 * Only continue if we have an active interface,
	 * either monitor or non-monitor should be present.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	NOTICE("Hardware button pressed, disabling radio.\n");

	rt61pci_disable_radio(rt2x00dev);
}

static void rt61pci_button_start(struct rt2x00_dev *rt2x00dev)
{
	struct rfkill *rfkill = &rt2x00dev->rfkill;

	/*
	 * Only start the button polling when
	 * the hardware button is present.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_SUPPORT_HW_BUTTON))
		return;

	/*
	 * Enable hardware button status reading.
	 */
	rt2x00_register_write(rt2x00dev, TEST_MODE_CSR, 0x00000200);

	rfkill->dev_name	= "rt61pci";
	rfkill->data		= (unsigned long)rt2x00dev;
	rfkill->poll		= rt61pci_button_poll;
	rfkill->enable_radio	= rt61pci_button_enable_radio;
	rfkill->disable_radio	= rt61pci_button_disable_radio;
	rfkill->current_status	= !!rt61pci_button_poll(rfkill->data);

	if (rfkill->current_status)
		SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);

	if (rfkill_add_device(rfkill))
		ERROR("Failed to register button handler.\n");
}

static void rt61pci_button_stop(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_SUPPORT_HW_BUTTON))
		return;

	rfkill_del_device(&rt2x00dev->rfkill);
}
#else /* CONFIG_RT61PCI_BUTTON */
static inline void rt61pci_button_start(struct rt2x00_dev *rt2x00dev)
{
	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW);
}

static inline void rt61pci_button_stop(struct rt2x00_dev *rt2x00dev){}
#endif /* CONFIG_RT61PCI_BUTTON */

/*
 * Configuration handlers.
 */
static void rt61pci_config_bssid(struct rt2x00_dev *rt2x00dev, u8 *bssid)
{
	u32 reg[2] = { 0, 0 };

	/*
	 * The BSSID is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 * We only need to set the BSS ID MASK at the correct offset.
	 */
	memcpy(&reg, bssid, ETH_ALEN);
	rt2x00_set_field32(&reg[1], MAC_CSR5_BSS_ID_MASK, 3);
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR4, &reg[0], sizeof(reg));
}

static void rt61pci_config_promisc(struct rt2x00_dev *rt2x00dev, int promisc)
{
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);

	if (promisc) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
}

static void rt61pci_config_type(struct rt2x00_dev *rt2x00dev, int type)
{
	u32 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (!(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED)) &&
	    !(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR)))
		return;

	rt2x00_register_write(rt2x00dev, TXRX_CSR9, 0);

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);

	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    (type == IEEE80211_IF_TYPE_IBSS || type == IEEE80211_IF_TYPE_STA))
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_TO_DS, 0);

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CRC, 1);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 0);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_PHYSICAL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_CONTROL, 1);
		rt2x00_set_field32(&reg, TXRX_CSR0_DROP_VERSION_ERROR, 1);
	}

	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_MULTICAST, 0);
	rt2x00_set_field32(&reg, TXRX_CSR0_DROP_BORADCAST, 0);

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt61pci_config_promisc(rt2x00dev, 1);

	/*
	 * Enable synchronisation.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR9, &reg);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_INTERVAL, 100 * 16);
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_TICKING, 1);
		rt2x00_set_field32(&reg, TXRX_CSR9_TBTT_ENABLE, 1);
	}
	if (type == IEEE80211_IF_TYPE_IBSS) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 2);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 1);
	} else if (type == IEEE80211_IF_TYPE_STA) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 1);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 0);
	} else if (type == IEEE80211_IF_TYPE_AP) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 0);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 1);
	} else if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
		   !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field32(&reg, TXRX_CSR9_TSF_SYNC_MODE, 0);
		rt2x00_set_field32(&reg, TXRX_CSR9_BEACON_GEN, 0);
	}
	rt2x00_register_write(rt2x00dev, TXRX_CSR9, reg);

	/*
	 * Change flags of enabled interfaces.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED);
	} else {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
	}
}

static void rt61pci_config_channel(struct rt2x00_dev *rt2x00dev,
	int rf2, int channel, int freq, int txpower)
{
	u8 reg = 0;
	u32 rf1 = 0;
	u32 rf3 = 0;
	u32 rf4 = 0;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00dev->rx_params.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00dev->tx_power;
	txpower = TXPOWER_TO_DEV(txpower);

	if (!GET_FLAG(rt2x00dev, CONFIG_RF_SEQUENCE) || channel <= 14)
		rf1 = 0x00002ccc;
	else if (channel == 36 ||
		(channel >= 100 && channel <= 116) ||
		channel >= 157)
		rf1 = 0x00002cd4;
	else
		rf1 = 0x00002cd0;

	if (channel <= 14) {
		rf3 = 0x00068455;
	} else if (!GET_FLAG(rt2x00dev, CONFIG_RF_SEQUENCE)) {
		if (channel >= 36 && channel <= 48)
			rf3 = 0x0009be55;
		else if (channel >= 52 && channel <= 64)
			rf3 = 0x0009ae55;
		else if (channel >= 100 && channel <= 112)
			rf3 = 0x000bae55;
		else
			rf3 = 0x000bbe55;
	} else {
		switch (channel) {
			case 36:
			case 40:
			case 44:
				rf3 = 0x00098455;
				break;
			case 48:
				rf3 = 0x00098655;
				break;
			case 52:
				rf3 = 0x00098855;
				break;
			case 56:
				rf3 = 0x00098c55;

			case 60:
				rf3 = 0x00098e55;
				break;
			case 64:
				rf3 = 0x00099255;
				break;
			case 100:
			case 104:
			case 108:
				rf3 = 0x000b9855;
				break;
			case 112:
			case 116:
			case 120:
			case 124:
				rf3 = 0x000b9a55;
				break;
			case 128:
			case 132:
				rf3 = 0x000b9c55;
				break;
			case 136:
			case 140:
				rf3 = 0x000b9e55;
				break;
			case 149:
			case 153:
			case 157:
			case 161:
			case 165:
				rf3 = 0x000ba255;
				break;
		}
	}

	if (channel < 14) {
		if (channel & 1)
			rf4 = 0x000ffa0b;
		else
			rf4 = 0x000ffa1f;
	} else if (channel == 14) {
		rf4 = 0x000ffa13;
	} else if (!GET_FLAG(rt2x00dev, CONFIG_RF_SEQUENCE)) {
		switch (channel) {
			case 36:
			case 56:
			case 116:
			case 136:
				rf4 = 0x000ffa23;
				break;
			case 40:
			case 60:
			case 100:
			case 120:
			case 140:
				rf4 = 0x000ffa03;
				break;
			case 44:
			case 64:
			case 104:
			case 124:
				rf4 = 0x000ffa0b;
				break;
			case 48:
			case 108:
			case 128:
				rf4 = 0x000ffa13;
				break;
			case 52:
			case 112:
			case 132:
				rf4 = 0x000ffa1b;
				break;
			case 149:
				rf4 = 0x000ffa1f;
				break;
			case 153:
				rf4 = 0x000ffa27;
				break;
			case 157:
				rf4 = 0x000ffa07;
				break;
			case 161:
				rf4 = 0x000ffa0f;
				break;
			case 165:
				rf4 = 0x000ffa17;
				break;
		}
	} else {
		switch (channel) {
			case 36:
			case 40:
			case 60:
			case 140:
			case 100:
			case 104:
			case 108:
			case 112:
			case 116:
			case 120:
				rf4 = 0x000c0a03;
				break;
			case 44:
			case 64:
			case 124:
			case 149:
				rf4 = 0x000c0a1b;
				break;
			case 48:
			case 128:
			case 153:
				rf4 = 0x000c0a0b;
				break;
			case 52:
			case 132:
				rf4 = 0x000c0a23;
				break;
			case 56:
			case 136:
				rf4 = 0x000c0a13;
				break;
			case 157:
			case 161:
			case 165:
				rf4 = 0x000c0a17;
				break;
		}
	}

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32(&rf3, RF3_TXPOWER, txpower);

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	/*
	 * Set Frequency offset.
	 */
	rt2x00_set_field32(&rf4, RF4_FREQ_OFFSET, rt2x00dev->freq_offset);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	udelay(200);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 | 0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	udelay(200);

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rf4);

	rt2x00_bbp_read(rt2x00dev, 3, &reg);
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2527))
		reg &= ~0x01;
	else
		reg |= 0x01;
	rt2x00_bbp_write(rt2x00dev, 3, reg);

	msleep(1);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.freq = freq;
	rt2x00dev->rx_params.channel = channel;

	rt2x00dev->tx_power = txpower;

	/*
	 * Update rf fields
	 */
	rt2x00dev->rf1 = rf1;
	rt2x00dev->rf2 = rf2;
	rt2x00dev->rf3 = rf3;
	rt2x00dev->rf4 = rf4;
}

static void rt61pci_config_txpower(struct rt2x00_dev *rt2x00dev, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00dev->tx_power)
		return;

	rt2x00_set_field32(&rt2x00dev->rf3, RF3_TXPOWER, txpower);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	udelay(200);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 | 0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	udelay(200);

	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf1);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf2);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3 & ~0x00000004);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf4);

	rt2x00dev->tx_power = txpower;
}

static void rt61pci_config_antenna(struct rt2x00_dev *rt2x00dev,
	int antenna, int phymode)
{
	u32 reg;
	u8 reg_r3;
	u8 reg_r4;
	u8 reg_r77;
	u8 frame_type;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.antenna == antenna)
		return;

	rt2x00_register_read(rt2x00dev, PHY_CSR0, &reg);

	if (phymode == MODE_IEEE80211A) {
		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA_A)) {
			rt2x00_bbp_write(rt2x00dev, 17, 0x38);
			rt2x00_bbp_write(rt2x00dev, 96, 0x78);
			rt2x00_bbp_write(rt2x00dev, 104, 0x48);
			rt2x00_bbp_write(rt2x00dev, 75, 0x80);
			rt2x00_bbp_write(rt2x00dev, 86, 0x80);
			rt2x00_bbp_write(rt2x00dev, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00dev, 17, 0x28);
			rt2x00_bbp_write(rt2x00dev, 96, 0x58);
			rt2x00_bbp_write(rt2x00dev, 104, 0x38);
			rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00dev, 35, 0x60);
		rt2x00_bbp_write(rt2x00dev, 97, 0x58);
		rt2x00_bbp_write(rt2x00dev, 98, 0x58);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 0);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 1);
	} else {
		if (GET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA_BG)) {
			rt2x00_bbp_write(rt2x00dev, 17, 0x30);
			rt2x00_bbp_write(rt2x00dev, 96, 0x68);
			rt2x00_bbp_write(rt2x00dev, 104, 0x3c);
			rt2x00_bbp_write(rt2x00dev, 75, 0x80);
			rt2x00_bbp_write(rt2x00dev, 86, 0x80);
			rt2x00_bbp_write(rt2x00dev, 88, 0x80);
		} else {
			rt2x00_bbp_write(rt2x00dev, 17, 0x20);
			rt2x00_bbp_write(rt2x00dev, 96, 0x48);
			rt2x00_bbp_write(rt2x00dev, 104, 0x2c);
			rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
			rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
		}
		rt2x00_bbp_write(rt2x00dev, 35, 0x50);
		rt2x00_bbp_write(rt2x00dev, 97, 0x48);
		rt2x00_bbp_write(rt2x00dev, 98, 0x48);

		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_BG, 1);
		rt2x00_set_field32(&reg, PHY_CSR0_PA_PE_A, 0);
	}

	rt2x00_register_write(rt2x00dev, PHY_CSR0, reg);

	rt2x00_bbp_read(rt2x00dev, 3, &reg_r3);
	rt2x00_bbp_read(rt2x00dev, 4, &reg_r4);
	rt2x00_bbp_read(rt2x00dev, 77, &reg_r77);

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF2527))
		reg_r3 &= ~0x01;
	reg_r4 &= ~0x23;
	frame_type = ~(GET_FLAG(rt2x00dev, CONFIG_FRAME_TYPE) << 5);

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
		if (antenna == 0) {
			/* Diversity. */
			reg_r4 |= 0x02;
			if (phymode != MODE_IEEE80211A)
				reg_r4 |= 0x20;
		} else if (antenna == 1) {
			/* RX: Antenna B */
			/* TX: Antenna A */
			reg_r4 |= 0x01;
			if (phymode == MODE_IEEE80211A)
				reg_r77 |= 0x03;
			else
				reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		} else if (antenna == 2) {
			/* RX: Antenna A */
			/* TX: Antenna B */
			reg_r4 |= 0x01;
			if (phymode == MODE_IEEE80211A)
				reg_r77 &= ~0x03;
			else
				reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		}
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2527) ||
		   (rt2x00_rf(&rt2x00dev->chip, RF2529) &&
		    GET_FLAG(rt2x00dev, CONFIG_DOUBLE_ANTENNA))) {
		if (antenna == 0) {
			/* Diversity. */
			reg_r4 |= 0x22;
			reg_r4 &= frame_type;
		} else if (antenna == 1) {
			/* RX: Antenna B */
			/* TX: Antenna A */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 &= ~0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		} else if (antenna == 2) {
			/* RX: Antenna A */
			/* TX: Antenna B */
			reg_r4 |= 0x21;
			reg_r4 &= frame_type;
			reg_r77 |= 0x03;
			rt2x00_bbp_write(rt2x00dev, 77, reg_r77);
		}
	}

	/*
	 * TODO: RF2529 with another antenna value than 2 are ignored.
	 * The legacy driver is unclear whether in those cases there is
	 * a possibility to switch antenna.
	 */

	rt2x00_bbp_write(rt2x00dev, 3, reg_r3);
	rt2x00_bbp_write(rt2x00dev, 4, reg_r4);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.antenna = antenna;
}

static void rt61pci_config_duration(struct rt2x00_dev *rt2x00dev,
	int short_slot_time)
{
	u32 reg;

	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_read(rt2x00dev, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_SLOT_TIME, short_slot_time);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR8, &reg);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS, SIFS);
	rt2x00_set_field32(&reg, MAC_CSR8_SIFS_AFTER_RX_OFDM, 3);
	rt2x00_set_field32(&reg, MAC_CSR8_EIFS, EIFS);
	rt2x00_register_write(rt2x00dev, MAC_CSR8, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_TSF_OFFSET, IEEE80211_HEADER);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_ENABLE, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);
}

static void rt61pci_config_rate(struct rt2x00_dev *rt2x00dev, const int rate)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
		pci_get_drvdata(rt2x00dev_pci(rt2x00dev)));
	u32 reg;
	u32 value;
	u32 preamble;

	preamble = DEVICE_GET_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	/*
	 * Extract the allowed ratemask from the device specific rate value,
	 * We need to set TXRX_CSR5 to the basic rate mask so we need to mask
	 * off the non-basic rates.
	 */
	reg = DEVICE_GET_RATE_FIELD(rate, RATEMASK) & DEV_BASIC_RATE;

	rt2x00_register_write(rt2x00dev, TXRX_CSR5, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	value = (conf->short_slot_time ? SHORT_DIFS :  DIFS) +
		PLCP + preamble + get_duration(ACK_SIZE, 10);
	rt2x00_set_field32(&reg, TXRX_CSR0_RX_ACK_TIMEOUT, value);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	if (preamble == SHORT_PREAMBLE)
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 1);
	else
		rt2x00_set_field32(&reg, TXRX_CSR4_AUTORESPOND_PREAMBLE, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);
}

static void rt61pci_config_phymode(struct rt2x00_dev *rt2x00dev,
	const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A &&
	    (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5325)))
		rate = &rt2x00dev->hw.modes[2].rates[
			rt2x00dev->hw.modes[2].num_rates - 1];
	else if (phymode == MODE_IEEE80211B)
		rate = &rt2x00dev->hw.modes[1].rates[
			rt2x00dev->hw.modes[1].num_rates - 1];
	else
		rate = &rt2x00dev->hw.modes[0].rates[
			rt2x00dev->hw.modes[0].num_rates - 1];

	rt61pci_config_rate(rt2x00dev, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00dev->rx_params.phymode = phymode;
}

static void rt61pci_config_mac_address(struct rt2x00_dev *rt2x00dev,
	void *addr)
{
	u32 reg[2] = { 0, 0 };

	/*
	 * The MAC address is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 * We only need to set the MAC_CSR3_UNICAST_TO_ME_MASK
	 * at the correct offset.
	 */
	memcpy(&reg, addr, ETH_ALEN);
	rt2x00_set_field32(&reg[1], MAC_CSR3_UNICAST_TO_ME_MASK, 0xff);
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR2, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt61pci_link_tuner(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;
	u32 reg;
	u32 rssi;
	u8 reg_r17;
	u8 up_bound;
	u8 low_bound;

	/*
	 * Retreive link quality.
	 */
	rssi = rt2x00_get_link(&rt2x00dev->link);
	if (!rssi)
		goto exit;


	/*
	 * Update LED.
	 */
	rt61pci_activity_led(rt2x00dev, rssi);

	/*
	 * Determine upper and lower limit for BBP17 register.
	 */
	if (rt2x00dev->rx_params.phymode == MODE_IEEE80211A) {
		up_bound = 0x48;
		low_bound = 0x28;
	} else {
		up_bound = 0x40;
		low_bound = 0x20;
	}

	rt2x00_bbp_read(rt2x00dev, 17, &reg_r17);

	if (rssi >= 85) {
		if (reg_r17 != 0x60)
			rt2x00_bbp_write(rt2x00dev, 17, 0x60);
		goto exit;
	} else if (rssi >= 62) {
		if (reg_r17 != up_bound)
			rt2x00_bbp_write(rt2x00dev, 17, up_bound);
		goto exit;
	} else if (rssi >= 54) {
		low_bound += 0x10;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00dev, 17, low_bound);
		goto exit;
	} else if (rssi >= 46) {
		low_bound += 0x08;
		if (reg_r17 != low_bound)
			rt2x00_bbp_write(rt2x00dev, 17, low_bound);
		goto exit;
	} else if (reg_r17 >= up_bound) {
		rt2x00_bbp_write(rt2x00dev, 17, up_bound);
		goto exit;
	}

	rt2x00_register_read(rt2x00dev, STA_CSR1, &reg);
	reg = rt2x00_get_field32(reg, STA_CSR1_FALSE_CCA_ERROR);

	if (reg > 512 && reg_r17 < up_bound)
		rt2x00_bbp_write(rt2x00dev, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > low_bound)
		rt2x00_bbp_write(rt2x00dev, 17, --reg_r17);

exit:
	queue_delayed_work(rt2x00dev->workqueue, &rt2x00dev->link.work,
		LINK_TUNE_INTERVAL);
}

/*
 * LED functions.
 */
static void rt61pci_enable_led(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;
	u16 led_reg;
	u8 arg0;
	u8 arg1;

	rt2x00_register_read(rt2x00dev, MAC_CSR14, &reg);
	rt2x00_set_field32(&reg, MAC_CSR14_ON_PERIOD, 70);
	rt2x00_set_field32(&reg, MAC_CSR14_OFF_PERIOD, 30);
	rt2x00_register_write(rt2x00dev, MAC_CSR14, reg);

	led_reg = rt2x00dev->led_reg;
	rt2x00_set_field16(&led_reg, MCU_LEDCS_RADIO_STATUS, 1);
	if (rt2x00dev->rx_params.phymode == MODE_IEEE80211A)
		rt2x00_set_field16(&led_reg, MCU_LEDCS_LINK_A_STATUS, 1);
	else
		rt2x00_set_field16(&led_reg, MCU_LEDCS_LINK_BG_STATUS, 1);

	arg0 = led_reg & 0xff;
	arg1 = (led_reg >> 8) & 0xff;

	rt2x00_mcu_request(rt2x00dev, MCU_LED, 0xff, arg0, arg1);
}

static void rt61pci_disable_led(struct rt2x00_dev *rt2x00dev)
{
	u16 led_reg;
	u8 arg0;
	u8 arg1;

	led_reg = rt2x00dev->led_reg;
	rt2x00_set_field16(&led_reg, MCU_LEDCS_RADIO_STATUS, 0);
	rt2x00_set_field16(&led_reg, MCU_LEDCS_LINK_BG_STATUS, 0);
	rt2x00_set_field16(&led_reg, MCU_LEDCS_LINK_A_STATUS, 0);

	arg0 = led_reg & 0xff;
	arg1 = (led_reg >> 8) & 0xff;

	rt2x00_mcu_request(rt2x00dev, MCU_LED, 0xff, arg0, arg1);
}

static void rt61pci_activity_led(struct rt2x00_dev *rt2x00dev, char rssi)
{
	u8 led;

	if (rt2x00dev->led_mode != LED_MODE_SIGNAL_STRENGTH)
		return;

	if (rssi <= 30)
		led = 0;
	else if (rssi <= 39)
		led = 1;
	else if (rssi <= 49)
		led = 2;
	else if (rssi <= 53)
		led = 3;
	else if (rssi <= 63)
		led = 4;
	else
		led = 5;

	rt2x00_mcu_request(rt2x00dev, MCU_LED_STRENGTH, 0xff, led, 0);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt61pci_set_state(struct rt2x00_dev *rt2x00dev,
	enum dev_state state)
{
	u32 reg;
	unsigned int i;
	char put_to_sleep;
	char current_state;

	put_to_sleep = (state != STATE_AWAKE);

	rt2x00_register_read(rt2x00dev, MAC_CSR12, &reg);
	rt2x00_set_field32(&reg, MAC_CSR12_FORCE_WAKEUP, !put_to_sleep);
	rt2x00_set_field32(&reg, MAC_CSR12_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00dev, MAC_CSR12, reg);

	if (put_to_sleep) {
		rt2x00_register_write(rt2x00dev, SOFT_RESET_CSR, 0x00000005);
		rt2x00_register_write(rt2x00dev, IO_CNTL_CSR, 0x0000001c);
		rt2x00_register_write(rt2x00dev, PCI_USEC_CSR, 0x00000060);
		rt2x00_mcu_request(rt2x00dev, MCU_SLEEP, 0xff, 0x00, 0x00);
	} else {
		rt2x00_register_write(rt2x00dev, SOFT_RESET_CSR, 0x00000007);
		rt2x00_register_write(rt2x00dev, IO_CNTL_CSR, 0x00000018);
		rt2x00_register_write(rt2x00dev, PCI_USEC_CSR, 0x00000020);
		rt2x00_mcu_request(rt2x00dev, MCU_WAKEUP, 0xff, 0x00, 0x00);
	}

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR12, &reg);
		current_state = rt2x00_get_field32(reg,
			MAC_CSR12_BBP_CURRENT_STATE);
		if (current_state == !put_to_sleep)
			return 0;
		msleep(10);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state %d.\n", !put_to_sleep, current_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt61pci_init_firmware_wait(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	for (i = 0; i < 150; i++) {
		if (GET_FLAG(rt2x00dev, FIRMWARE_FAILED))
			return -EIO;
		if (GET_FLAG(rt2x00dev, FIRMWARE_LOADED))
			return 0;
		msleep(20);
	}

	ERROR("Firmware loading timed out.\n");
	return -EIO;
}

static void rt61pci_init_firmware_cont(const struct firmware *fw,
	void *context)
{
	struct rt2x00_dev *rt2x00dev = context;
	int i;
	u32 reg;
	u16 crc;

	if (!fw || !fw->size || !fw->data) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	/*
	 * Wait for stable hardware.
	 */
	for (i = 0; i < 100; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
		if (reg)
			break;
		msleep(1);
	}

	if (!reg) {
		ERROR("Unstable hardware.\n");
		goto exit;
	}

	/*
	 * Prepare MCU and mailbox for firmware loading.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MCU_CNTL_CSR_RESET, 1);
	rt2x00_register_write(rt2x00dev, MCU_CNTL_CSR, reg);
	rt2x00_register_write(rt2x00dev, M2H_CMD_DONE_CSR, 0xffffffff);
	rt2x00_register_write(rt2x00dev, H2M_MAILBOX_CSR, 0);
	rt2x00_register_write(rt2x00dev, HOST_CMD_CSR, 0);

	/*
	 * Validate the firmware using 16 bit CRC.
	 * The last 2 bytes of the firmware are the CRC
	 * so substract those 2 bytes from the CRC checksum,
	 * and set those 2 bytes to 0 when calculating CRC.
	 */
	reg = 0;
	crc = crc_itu_t(0, fw->data, fw->size - 2);
	crc = crc_itu_t(crc, (u8*)&reg, 2);

	if (crc != (fw->data[fw->size - 2] << 8 | fw->data[fw->size - 1])) {
		ERROR("Firmware CRC error.\n");
		goto exit;
	}

	rt2x00_set_chip_fw(&rt2x00dev->chip,
		fw->data[fw->size - 4], fw->data[fw->size - 3]);

	/*
	 * Write firmware to device.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MCU_CNTL_CSR_RESET, 1);
	rt2x00_set_field32(&reg, MCU_CNTL_CSR_SELECT_BANK, 1);
	rt2x00_register_write(rt2x00dev, MCU_CNTL_CSR, reg);

	rt2x00_register_multiwrite(
		rt2x00dev, FIRMWARE_IMAGE_BASE, (u32*)fw->data, fw->size);

	rt2x00_set_field32(&reg, MCU_CNTL_CSR_SELECT_BANK, 0);
	rt2x00_register_write(rt2x00dev, MCU_CNTL_CSR, reg);

	rt2x00_set_field32(&reg, MCU_CNTL_CSR_RESET, 0);
	rt2x00_register_write(rt2x00dev, MCU_CNTL_CSR, reg);

	for (i = 0; i < 100; i++) {
		rt2x00_register_read(rt2x00dev, MCU_CNTL_CSR, &reg);
		if (rt2x00_get_field32(reg, MCU_CNTL_CSR_READY))
			break;
		msleep(1);
	}

	if (i == 100) {
		ERROR("MCU Control register not ready.\n");
		goto exit;
	}

	/*
	 * Reset MAC and BBP registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	SET_FLAG(rt2x00dev, FIRMWARE_LOADED);

	return;

exit:
	SET_FLAG(rt2x00dev, FIRMWARE_FAILED);
}

static int rt61pci_init_firmware(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;
	static const struct {
		char	*name;
		u32	chip;
	} firmware[] = {
		{ "rt2561.bin",		RT2561 },
		{ "rt2561s.bin",	RT2561s },
		{ "rt2661.bin",		RT2661 },
	};

	/*
	 * Read correct firmware from harddisk.
	 */
	for (i = 0; i < ARRAY_SIZE(firmware); i++) {
		if (!rt2x00_rt(&rt2x00dev->chip, firmware[i].chip))
			continue;
		return request_firmware_nowait(THIS_MODULE, 1,
			firmware[i].name, &rt2x00dev_pci(rt2x00dev)->dev,
			rt2x00dev, rt61pci_init_firmware_cont);
	}

	return -EINVAL;
}

static int rt61pci_alloc_dma_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type, void (*handler)(void *),
	const u16 max_entries, const u16 data_size, const u16 desc_size)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kmalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = pci_alloc_consistent(rt2x00dev_pci(rt2x00dev),
		rt2x00_get_ring_size(ring), &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].ring = ring;
		ring->entry[i].skb = NULL;
		ring->entry[i].priv = ring->data_addr
			+ (i * ring->desc_size);
		ring->entry[i].data_addr = ring->data_addr
			+ (ring->stats.limit * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (ring->stats.limit * ring->desc_size)
			+ (i * ring->data_size);
	}

	return 0;
}

static void rt61pci_free_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];

	if (ring->data_addr)
		pci_free_consistent(rt2x00dev_pci(rt2x00dev),
			rt2x00_get_ring_size(ring),
			ring->data_addr, ring->data_dma);
	ring->data_addr = NULL;

	kfree(ring->entry);
	ring->entry = NULL;
}

static int rt61pci_allocate_dma_rings(struct rt2x00_dev *rt2x00dev)
{
	if (rt61pci_alloc_dma_ring(rt2x00dev, RING_RX,
		rt61pci_rxdone, RX_ENTRIES, DATA_FRAME_SIZE,
		RXD_DESC_SIZE) ||
	    rt61pci_alloc_dma_ring(rt2x00dev, RING_AC_BK,
		rt61pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt61pci_alloc_dma_ring(rt2x00dev, RING_AC_BE,
		rt61pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt61pci_alloc_dma_ring(rt2x00dev, RING_AC_VI,
		rt61pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt61pci_alloc_dma_ring(rt2x00dev, RING_AC_VO,
		rt61pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt61pci_alloc_dma_ring(rt2x00dev, RING_PRIO,
		rt61pci_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt61pci_alloc_dma_ring(rt2x00dev, RING_BEACON,
		rt61pci_beacondone, BEACON_ENTRIES, MGMT_FRAME_SIZE,
		TXD_DESC_SIZE) ) {
		return -ENOMEM;
	}

	return 0;
}

static void rt61pci_free_rings(struct rt2x00_dev *rt2x00dev)
{
	rt61pci_free_ring(rt2x00dev, RING_RX);
	rt61pci_free_ring(rt2x00dev, RING_AC_BK);
	rt61pci_free_ring(rt2x00dev, RING_AC_BE);
	rt61pci_free_ring(rt2x00dev, RING_AC_VI);
	rt61pci_free_ring(rt2x00dev, RING_AC_VO);
	rt61pci_free_ring(rt2x00dev, RING_PRIO);
	rt61pci_free_ring(rt2x00dev, RING_BEACON);
}

static void rt61pci_init_rxring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	void *rxd;
	unsigned int i;
	u32 word;

	memset(ring->data_addr, 0x00, rt2x00_get_ring_size(ring));

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		rxd = rt2x00pci_desc_addr(&ring->entry[i]);

		rt2x00_desc_read(rxd, 5, &word);
		rt2x00_set_field32(&word, RXD_W5_BUFFER_PHYSICAL_ADDRESS,
			ring->entry[i].data_dma);
		rt2x00_desc_write(rxd, 5, word);

		rt2x00_desc_read(rxd, 0, &word);
		rt2x00_set_field32(&word, RXD_W0_OWNER_NIC, 1);
		rt2x00_desc_write(rxd, 0, word);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt61pci_init_txring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	void *txd;
	unsigned int i;
	u32 word;

	memset(ring->data_addr, 0x00, rt2x00_get_ring_size(ring));

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		txd = rt2x00pci_desc_addr(&ring->entry[i]);

		rt2x00_desc_read(txd, 1, &word);
		rt2x00_set_field32(&word, TXD_W1_BUFFER_COUNT, 1);
		rt2x00_desc_write(txd, 1, word);

		rt2x00_desc_read(txd, 5, &word);
		rt2x00_set_field32(&word, TXD_W5_PID_TYPE, ring_type);
		rt2x00_set_field32(&word, TXD_W5_PID_SUBTYPE, i);
		rt2x00_desc_write(txd, 5, word);

		rt2x00_desc_read(txd, 6, &word);
		rt2x00_set_field32(&word, TXD_W6_BUFFER_PHYSICAL_ADDRESS,
			ring->entry[i].data_dma);
		rt2x00_desc_write(txd, 6, word);

		rt2x00_desc_read(txd, 0, &word);
		rt2x00_set_field32(&word, TXD_W0_VALID, 0);
		rt2x00_set_field32(&word, TXD_W0_OWNER_NIC, 0);
		rt2x00_desc_write(txd, 0, word);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt61pci_init_rings(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	/*
	 * Initialize rings.
	 */
	rt61pci_init_rxring(rt2x00dev, RING_RX);
	rt61pci_init_txring(rt2x00dev, RING_AC_BK);
	rt61pci_init_txring(rt2x00dev, RING_AC_BE);
	rt61pci_init_txring(rt2x00dev, RING_AC_VI);
	rt61pci_init_txring(rt2x00dev, RING_AC_VO);
	rt61pci_init_txring(rt2x00dev, RING_PRIO);
	rt61pci_init_txring(rt2x00dev, RING_BEACON);

	/*
	 * Initialize registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, TX_RING_CSR0_AC0_RING_SIZE,
		rt2x00dev->ring[RING_AC_BK].stats.limit);
	rt2x00_set_field32(&reg, TX_RING_CSR0_AC1_RING_SIZE,
		rt2x00dev->ring[RING_AC_BE].stats.limit);
	rt2x00_set_field32(&reg, TX_RING_CSR0_AC2_RING_SIZE,
		rt2x00dev->ring[RING_AC_VI].stats.limit);
	rt2x00_set_field32(&reg, TX_RING_CSR0_AC3_RING_SIZE,
		rt2x00dev->ring[RING_AC_VO].stats.limit);
	rt2x00_register_write(rt2x00dev, TX_RING_CSR0, reg);

	reg = 0;
	rt2x00_set_field32(&reg, TX_RING_CSR1_MGMT_RING_SIZE,
		rt2x00dev->ring[RING_PRIO].stats.limit);
	rt2x00_set_field32(&reg, TX_RING_CSR1_TXD_SIZE,
		rt2x00dev->ring[RING_AC_BK].desc_size / 4);
	rt2x00_register_write(rt2x00dev, TX_RING_CSR1, reg);

	reg = 0;
	rt2x00_set_field32(&reg, AC0_BASE_CSR_RING_REGISTER,
		rt2x00dev->ring[RING_AC_BK].data_dma);
	rt2x00_register_write(rt2x00dev, AC0_BASE_CSR, reg);

	reg = 0;
	rt2x00_set_field32(&reg, AC1_BASE_CSR_RING_REGISTER,
		rt2x00dev->ring[RING_AC_BE].data_dma);
	rt2x00_register_write(rt2x00dev, AC1_BASE_CSR, reg);

	reg = 0;
	rt2x00_set_field32(&reg, AC2_BASE_CSR_RING_REGISTER,
		rt2x00dev->ring[RING_AC_VI].data_dma);
	rt2x00_register_write(rt2x00dev, AC2_BASE_CSR, reg);

	reg = 0;
	rt2x00_set_field32(&reg, AC3_BASE_CSR_RING_REGISTER,
		rt2x00dev->ring[RING_AC_VO].data_dma);
	rt2x00_register_write(rt2x00dev, AC3_BASE_CSR, reg);

	reg = 0;
	rt2x00_set_field32(&reg, MGMT_BASE_CSR_RING_REGISTER,
		rt2x00dev->ring[RING_PRIO].data_dma);
	rt2x00_register_write(rt2x00dev, MGMT_BASE_CSR, reg);

	reg = 0;
	rt2x00_set_field32(&reg, RX_RING_CSR_RING_SIZE,
		rt2x00dev->ring[RING_RX].stats.limit);
	rt2x00_set_field32(&reg, RX_RING_CSR_RXD_SIZE,
		rt2x00dev->ring[RING_RX].desc_size / 4);
	rt2x00_set_field32(&reg, RX_RING_CSR_RXD_WRITEBACK_SIZE, 4);
	rt2x00_register_write(rt2x00dev, RX_RING_CSR, reg);

	reg = 0;
	rt2x00_set_field32(&reg, RX_BASE_CSR_RING_REGISTER,
		rt2x00dev->ring[RING_RX].data_dma);
	rt2x00_register_write(rt2x00dev, RX_BASE_CSR, reg);

	rt2x00_register_write(rt2x00dev, TX_DMA_DST_CSR, 0x000000aa);
	rt2x00_register_write(rt2x00dev, LOAD_TX_RING_CSR, 0x0000001f);
	rt2x00_register_write(rt2x00dev, RX_CNTL_CSR, 0x00000002);

	return 0;
}

static int rt61pci_init_registers(struct rt2x00_dev *rt2x00dev)
{
	u32 reg;

	if (rt61pci_set_state(rt2x00dev, STATE_AWAKE))
		return -EBUSY;

	rt2x00_register_write(rt2x00dev, MAC_CSR10, 0x00000718);

	rt2x00_register_write(rt2x00dev, TXRX_CSR0, 0x025eb032);

	rt2x00_register_write(rt2x00dev, TXRX_CSR1, 0x9eb39eb3);
	rt2x00_register_write(rt2x00dev, TXRX_CSR2, 0x8a8b8c8d);
	rt2x00_register_write(rt2x00dev, TXRX_CSR3, 0x00858687);

	rt2x00_register_write(rt2x00dev, TXRX_CSR7, 0x2e31353b);
	rt2x00_register_write(rt2x00dev, TXRX_CSR8, 0x2a2a2a2c);

	rt2x00_register_write(rt2x00dev, TXRX_CSR15, 0x0000000f);

	rt2x00_register_write(rt2x00dev, MAC_CSR6, 0x00000fff);

	rt2x00_register_write(rt2x00dev, MAC_CSR13, 0x0000e000);

	rt2x00_register_write(rt2x00dev, SEC_CSR0, 0x00000000);
	rt2x00_register_write(rt2x00dev, SEC_CSR1, 0x00000000);
	rt2x00_register_write(rt2x00dev, SEC_CSR5, 0x00000000);

	rt2x00_register_read(rt2x00dev, AC_TXOP_CSR0, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC0_TX_OP, 0);
	rt2x00_set_field32(&reg, AC_TXOP_CSR0_AC1_TX_OP, 0);
	rt2x00_register_write(rt2x00dev, AC_TXOP_CSR0, reg);

	rt2x00_register_read(rt2x00dev, AC_TXOP_CSR1, &reg);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC2_TX_OP, 192);
	rt2x00_set_field32(&reg, AC_TXOP_CSR1_AC3_TX_OP, 48);
	rt2x00_register_write(rt2x00dev, AC_TXOP_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR9, &reg);
	rt2x00_set_field32(&reg, MAC_CSR9_CW_SELECT, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_AUTO_TX_SEQ, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_write(rt2x00dev, PHY_CSR1, 0x000023b0);
	rt2x00_register_write(rt2x00dev, PHY_CSR5, 0x060a100c);
	rt2x00_register_write(rt2x00dev, PHY_CSR6, 0x00080606);
	rt2x00_register_write(rt2x00dev, PHY_CSR7, 0x00000a08);

	rt2x00_register_write(rt2x00dev, PCI_CFG_CSR, 0x28ca4404);

	rt2x00_register_write(rt2x00dev, M2H_CMD_DONE_CSR, 0xffffffff);

	/*
	 * We must clear the error counters.
	 * These registers are cleared on read,
	 * so we may pass a useless variable to store the value.
	 */
	rt2x00_register_read(rt2x00dev, STA_CSR0, &reg);
	rt2x00_register_read(rt2x00dev, STA_CSR1, &reg);
	rt2x00_register_read(rt2x00dev, STA_CSR2, &reg);

	/*
	 * Reset MAC and BBP registers.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 1);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_SOFT_RESET, 0);
	rt2x00_set_field32(&reg, MAC_CSR1_BBP_RESET, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR1, &reg);
	rt2x00_set_field32(&reg, MAC_CSR1_HOST_READY, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, reg);

	return 0;
}

static int rt61pci_init_bbp(struct rt2x00_dev *rt2x00dev)
{
	u8 reg_id;
	u8 value;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00dev, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00dev, 3, 0x00);
	rt2x00_bbp_write(rt2x00dev, 15, 0x30);
	rt2x00_bbp_write(rt2x00dev, 17, 0x20);
	rt2x00_bbp_write(rt2x00dev, 21, 0xc8);
	rt2x00_bbp_write(rt2x00dev, 22, 0x38);
	rt2x00_bbp_write(rt2x00dev, 23, 0x06);
	rt2x00_bbp_write(rt2x00dev, 24, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 25, 0x0a);
	rt2x00_bbp_write(rt2x00dev, 26, 0x0d);
	rt2x00_bbp_write(rt2x00dev, 34, 0x12);
	rt2x00_bbp_write(rt2x00dev, 37, 0x07);
	rt2x00_bbp_write(rt2x00dev, 39, 0xf8);
	rt2x00_bbp_write(rt2x00dev, 41, 0x60);
	rt2x00_bbp_write(rt2x00dev, 53, 0x10);
	rt2x00_bbp_write(rt2x00dev, 54, 0x18);
	rt2x00_bbp_write(rt2x00dev, 60, 0x10);
	rt2x00_bbp_write(rt2x00dev, 61, 0x04);
	rt2x00_bbp_write(rt2x00dev, 62, 0x04);
	rt2x00_bbp_write(rt2x00dev, 75, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 86, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 88, 0xfe);
	rt2x00_bbp_write(rt2x00dev, 90, 0x0f);
	rt2x00_bbp_write(rt2x00dev, 99, 0x00);
	rt2x00_bbp_write(rt2x00dev, 102, 0x16);
	rt2x00_bbp_write(rt2x00dev, 107, 0x04);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		if (rt2x00dev->eeprom[i] != 0xffff &&
		    rt2x00dev->eeprom[i] != 0x0000) {
			reg_id = rt2x00_get_field16(
				rt2x00dev->eeprom[i], EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(
				rt2x00dev->eeprom[i], EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00dev, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt61pci_init_channel_time(struct rt2x00_dev *rt2x00dev)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Only initialize the channel_change_time
	 * if it has not been set previously.
	 */
	if (rt2x00dev->hw.channel_change_time)
		return 0;

	/*
	 * Invalidate the rx_params.channel value to make sure
	 * the config channel will be correctly executed.
	 */
	rt2x00dev->rx_params.channel = 0;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt61pci_config_channel(rt2x00dev,
		rt2x00dev->hw.modes[0].channels[0].val,
		rt2x00dev->hw.modes[0].channels[0].chan,
		rt2x00dev->hw.modes[0].channels[0].freq,
		rt2x00dev->hw.modes[0].channels[0].power_level);
	jiffies_end = jiffies;

	rt2x00dev->hw.channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00dev->hw.channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt61pci_initialize(struct rt2x00_dev *rt2x00dev)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00dev_pci(rt2x00dev));

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return 0;

	/*
	 * We must wait on the firmware before
	 * we can safely continue.
	 */
	if (rt61pci_init_firmware_wait(rt2x00dev))
		return -ENODEV;

	/*
	 * Reset the channel_change_time value
	 * to make sure it will be correctly initialized
	 * after the radio has been enabled.
	 */
	rt2x00dev->hw.channel_change_time = 0;

	/*
	 * Allocate all data rings.
	 */
	if (rt61pci_allocate_dma_rings(rt2x00dev)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Register interrupt handler.
	 */
	if (request_irq(rt2x00dev_pci(rt2x00dev)->irq, rt61pci_interrupt,
		IRQF_SHARED, net_dev->name, rt2x00dev)) {
		ERROR("IRQ %d allocation failed.\n",
			rt2x00dev_pci(rt2x00dev)->irq);
		goto exit_fail;
	}

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt61pci_free_rings(rt2x00dev);

	return -EIO;
}

static void rt61pci_uninitialize(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00dev->scan)
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_CANCELLED);

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00dev->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt61pci_free_rings(rt2x00dev);

	/*
	 * Free irq line.
	 */
	free_irq(rt2x00dev_pci(rt2x00dev)->irq, rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt61pci_enable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00dev_pci(rt2x00dev));
	u32 reg;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO) ||
	    !GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW))
		return 0;

	/*
	 * Initialize all registers.
	 */
	if (rt61pci_init_rings(rt2x00dev) ||
	    rt61pci_init_registers(rt2x00dev) ||
	    rt61pci_init_bbp(rt2x00dev)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt61pci_init_channel_time(rt2x00dev))
		goto exit_fail;

	/*
	 * Clear interrupts.
	 */
	rt2x00_register_read(rt2x00dev, INT_SOURCE_CSR, &reg);
	rt2x00_register_write(rt2x00dev, INT_SOURCE_CSR, reg);

	rt2x00_register_read(rt2x00dev, MCU_INT_SOURCE_CSR, &reg);
	rt2x00_register_write(rt2x00dev, MCU_INT_SOURCE_CSR, reg);

	/*
	 * Enable interrupts.
	 */
	reg = 0;
	rt2x00_set_field32(&reg, INT_MASK_CSR_TX_ABORT_DONE, 1);
	rt2x00_set_field32(&reg, INT_MASK_CSR_MITIGATION_PERIOD, 0xff);
	rt2x00_register_write(rt2x00dev, INT_MASK_CSR, reg);

	rt2x00_register_write(rt2x00dev, MCU_INT_MASK_CSR, 0x00000000);

	/*
	 * Enable RX.
	 */
	rt2x00_register_write(rt2x00dev, RX_CNTL_CSR, 0x00000001);
	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * Enable LED
	 */
	rt61pci_enable_led(rt2x00dev);

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	ieee80211_start_queues(net_dev);
	ieee80211_netif_oper(net_dev, NETIF_WAKE);

	return 0;

exit_fail:
	rt61pci_uninitialize(rt2x00dev);
	return -EIO;
}

static void rt61pci_disable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00dev_pci(rt2x00dev));
	u32 reg;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	ieee80211_netif_oper(net_dev, NETIF_STOP);
	ieee80211_stop_queues(net_dev);

	/*
	 * Disable LED
	 */
	rt61pci_disable_led(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	rt2x00_register_write(rt2x00dev, MAC_CSR10, 0x00001818);

	/*
	 * Disable synchronisation.
	 */
	rt2x00_register_write(rt2x00dev, TXRX_CSR9, 0);

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00dev, TX_CNTL_CSR, &reg);
	rt2x00_set_field32(&reg, TX_CNTL_CSR_ABORT_TX_AC0, 1);
	rt2x00_set_field32(&reg, TX_CNTL_CSR_ABORT_TX_AC1, 1);
	rt2x00_set_field32(&reg, TX_CNTL_CSR_ABORT_TX_AC2, 1);
	rt2x00_set_field32(&reg, TX_CNTL_CSR_ABORT_TX_AC3, 1);
	rt2x00_set_field32(&reg, TX_CNTL_CSR_ABORT_TX_MGMT, 1);
	rt2x00_register_write(rt2x00dev, TX_CNTL_CSR, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	/*
	 * Disable interrupts.
	 */
	reg = 0xffffffff;
	rt2x00_set_field32(&reg, INT_MASK_CSR_ENABLE_MITIGATION, 0);
	rt2x00_register_write(rt2x00dev, INT_MASK_CSR, reg);

	rt2x00_register_write(rt2x00dev, MCU_INT_MASK_CSR, 0xffffffff);
}

/*
 * RTS frame creation.
 */
static struct sk_buff* rt61pci_create_rts(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_hdr *hdr, unsigned short duration)
{
	struct ieee80211_hdr *ieee80211hdr;
	struct sk_buff *skb;
	u16 frame_control;

	skb = dev_alloc_skb(IEEE80211_HEADER);
	if (!skb)
		return NULL;

	/*
	 * Copy the entire header over to RTS frame.
	 */
	memcpy(skb_put(skb, IEEE80211_HEADER), hdr, IEEE80211_HEADER);
	ieee80211hdr = (struct ieee80211_hdr*)skb->data;

	frame_control = IEEE80211_FTYPE_CTL | IEEE80211_STYPE_RTS;
	ieee80211hdr->frame_control = cpu_to_le16(frame_control);

	ieee80211hdr->duration_id += cpu_to_le16(duration);

	ieee80211hdr->seq_ctrl = 0;

	return skb;
}

/*
 * TX descriptor initialization
 */
static void rt61pci_write_tx_desc(struct rt2x00_dev *rt2x00dev,
	void *txd, struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	int tx_rate;
	u32 word;
	u32 length;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 frame_control;
	u16 seq_ctrl;
	char rts_frame;
	char ofdm_rate;
	char req_timestamp;
	char more_frag;
	char req_seq;
	char ifs;
	char queue;
	u8 signal;
	u8 service;
	u8 bitrate;

	/*
	 * We require the ring structure this packet is being send to.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring))
		return;

	/*
	 * Read required fields from ieee80211 header.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	seq_ctrl = le16_to_cpu(ieee80211hdr->seq_ctrl);

	/*
	 * Check if this frame is a RTS frame.
	 */
	rts_frame = !!(((frame_control & IEEE80211_FCTL_FTYPE) ==
		        IEEE80211_FTYPE_CTL) &&
		       ((frame_control & IEEE80211_FCTL_STYPE) ==
		        IEEE80211_STYPE_RTS));

	/*
	 * Check which rate should be used for this frame.
	 */
	if (rts_frame && control->rts_cts_rate)
		tx_rate = control->rts_cts_rate;
	else
		tx_rate = control->tx_rate;

	/*
	 * Are we working with OFDM rates.
	 */
	ofdm_rate = !!(DEVICE_GET_RATE_FIELD(tx_rate, RATEMASK) &
		       DEV_OFDM_RATE);

	/*
	 * Check if more fragments will follow this frame.
	 */
	more_frag = !!(ieee80211_get_morefrag(ieee80211hdr));

	/*
	 * Check if we require to enable the hw sequence counter.
	 */
	req_seq = !!(rt2x00_require_sequence(ieee80211hdr));

	/*
	 * Beacons and probe responses require the tsf timestamp
	 * to be inserted into the frame.
	 */
	req_timestamp = !!(control->queue == IEEE80211_TX_QUEUE_BEACON ||
		  	   control->pkt_type == PKT_PROBE_RESP);

	/*
	 * Determine with what IFS priority this frame should be send.
	 * Set ifs to IFS_SIFS when the this is not the first fragment,
	 * or this fragment came after RTS/CTS.
	 */
	if (((seq_ctrl & IEEE80211_SCTL_FRAG) > 0) || rts_frame)
		ifs = IFS_SIFS;
	else
		ifs = IFS_BACKOFF;

	/*
	 * Determine queue identification number.
	 */
	if (control->queue == IEEE80211_TX_QUEUE_BEACON)
		queue = 15;
	else if (control->queue == IEEE80211_TX_QUEUE_SVP)
		queue = 13;
	else
		queue = control->queue;

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * How the length should be processed depends
	 * on if we are working with OFDM rates or not.
	 */
	if (ofdm_rate) {
		residual = 0;
		length_high = (length >> 6) & 0x3f;
		length_low = (length & 0x3f);

	} else {
		bitrate = DEVICE_GET_RATE_FIELD(tx_rate, RATE);

		/*
		 * Convert length to microseconds.
		 */
		residual = get_duration_res(length, bitrate);
		length = get_duration(length, bitrate);

		if (residual != 0)
			length++;

		length_high = length >> 8;
		length_low = length & 0xff;
	}

	/*
	 * Create the signal and service values.
	 */
	signal = DEVICE_GET_RATE_FIELD(tx_rate, PLCP);
	if (DEVICE_GET_RATE_FIELD(tx_rate, PREAMBLE))
		signal |= 0x08;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	/*
	 * Start writing the descriptor words.
	 */
	rt2x00_desc_read(txd, 1, &word);
	rt2x00_set_field32(&word, TXD_W1_HOST_Q_ID, queue);
	rt2x00_set_field32(&word, TXD_W1_AIFSN, ring->tx_params.aifs);
	rt2x00_set_field32(&word, TXD_W1_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&word, TXD_W1_CWMAX, ring->tx_params.cw_max);
	rt2x00_set_field32(&word, TXD_W1_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&word, TXD_W1_HW_SEQUENCE, req_seq);
	rt2x00_desc_write(txd, 1, word);

	rt2x00_desc_read(txd, 2, &word);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SERVICE, service);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_HIGH, length_high);
	rt2x00_desc_write(txd, 2, word);

	rt2x00_desc_read(txd, 5, &word);
	rt2x00_set_field32(&word, TXD_W5_TX_POWER,
		TXPOWER_TO_DEV(control->power_level));
	rt2x00_set_field32(&word, TXD_W5_WAITING_DMA_DONE_INT, 1);
	rt2x00_desc_write(txd, 5, word);

	rt2x00_desc_read(txd, 11, &word);
	rt2x00_set_field32(&word, TXD_W11_BUFFER_LENGTH0, skb->len);
	rt2x00_desc_write(txd, 11, word);

	rt2x00_desc_read(txd, 0, &word);
	rt2x00_set_field32(&word, TXD_W0_OWNER_NIC, 1);
	rt2x00_set_field32(&word, TXD_W0_VALID, 1);
	rt2x00_set_field32(&word, TXD_W0_MORE_FRAG, more_frag);
	rt2x00_set_field32(&word, TXD_W0_ACK, !control->no_ack);
	rt2x00_set_field32(&word, TXD_W0_TIMESTAMP, req_timestamp);
	rt2x00_set_field32(&word, TXD_W0_OFDM, ofdm_rate);
	rt2x00_set_field32(&word, TXD_W0_IFS, ifs);
	rt2x00_set_field32(&word, TXD_W0_RETRY_MODE, 0);
	rt2x00_set_field32(&word, TXD_W0_TKIP_MIC, 0);
	rt2x00_set_field32(&word, TXD_W0_DATABYTE_COUNT, skb->len);
	rt2x00_set_field32(&word, TXD_W0_CIPHER_ALG, CIPHER_NONE);
	rt2x00_desc_write(txd, 0, word);
}

/*
 * Interrupt functions.
 */
static void rt61pci_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry = rt2x00_get_data_entry(
		&rt2x00dev->ring[RING_BEACON]);
	struct sk_buff *skb;

	skb = ieee80211_beacon_get(ring->net_dev,
		rt2x00dev->interface.id, &entry->tx_status.control);
	if (!skb)
		return;

	rt61pci_beacon_update(ring->net_dev, skb, &entry->tx_status.control);

	dev_kfree_skb_any(skb);
}

static void rt61pci_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(ring->net_dev);
	struct data_entry *entry;
	struct sk_buff *skb;
	void *rxd;
	u32 word0;
	u32 word1;
	u16 size;
	char total_packet;
	char total_rssi;

	/*
	 * Initialize variable for average RSSI calculation.
	 */
	total_packet = 0;
	total_rssi = 0;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00pci_desc_addr(entry);
		rt2x00_desc_read(rxd, 0, &word0);
		rt2x00_desc_read(rxd, 1, &word1);

		if (rt2x00_get_field32(word0, RXD_W0_OWNER_NIC))
			break;

		size = rt2x00_get_field32(word0, RXD_W0_DATABYTE_COUNT);

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(word0, RXD_W0_CRC)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				return;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size), rt2x00pci_data_addr(entry),
				size);

			rt2x00dev->rx_params.rate = device_signal_to_rate(
				&rt2x00dev->hw.modes[1],
				rt2x00_get_field32(word1, RXD_W1_SIGNAL),
				rt2x00_get_field32(word0, RXD_W0_OFDM));

			rt2x00dev->rx_params.ssi =
				rt2x00_get_field32(word1, RXD_W1_RSSI);

			__ieee80211_rx(ring->net_dev,
				skb, &rt2x00dev->rx_params);

			total_packet++;
			total_rssi += rt2x00dev->rx_params.ssi;
		}
		rt2x00_set_field32(&word0, RXD_W0_OWNER_NIC, 1);
		rt2x00_desc_write(rxd, 0, word0);
		rt2x00_ring_index_inc(ring);
	}

	/*
	 * Update link statistics
	 */
	rt2x00_update_link(&rt2x00dev->link, total_packet, total_rssi);
}

static void rt61pci_txdone_entry(struct data_entry *entry, u32 sta_csr4)
{
	struct rt2x00_dev *rt2x00dev =
		ieee80211_dev_hw_data(entry->ring->net_dev);
	void *txd;
	u32 word;
	int tx_status;
	int ack;
	int ring_full;

	txd = rt2x00pci_desc_addr(entry);
	rt2x00_desc_read(txd, 0, &word);

	if (rt2x00_get_field32(word, TXD_W0_OWNER_NIC) ||
	    !rt2x00_get_field32(word, TXD_W0_VALID))
		return;

	ack = rt2x00_get_field32(word, TXD_W0_ACK);

	/*
	 * TODO: How can te below field be set correctly?
	 */
	entry->tx_status.tx_filtered = 0;

	entry->tx_status.queue_length = entry->ring->stats.limit;
	entry->tx_status.queue_number = entry->tx_status.control.queue;

	/*
	 * The TXD_W0_RESULT field will only be set when
	 * we had requested an ACK. So we have received an
	 * ACK response when ACK was requested and status
	 * was succesfull.
	 */
	tx_status = rt2x00_get_field32(sta_csr4, STA_CSR4_TX_RESULT);
	entry->tx_status.ack = 0;
	entry->tx_status.excessive_retries = 0;
	if (ack && (tx_status == TX_SUCCESS ||
	    tx_status == TX_SUCCESS_RETRY))
		entry->tx_status.ack = 1;
	else if (ack && tx_status == TX_FAIL_RETRY) {
		rt2x00dev->low_level_stats.dot11ACKFailureCount++;
		entry->tx_status.excessive_retries = 1;
	}

	rt2x00_bbp_read(rt2x00dev, 32,
		(u8*)&entry->tx_status.ack_signal);

	entry->tx_status.retry_count = rt2x00_get_field32(
		sta_csr4, STA_CSR4_RETRY_COUNT);

	if (!GET_FLAG(entry, ENTRY_RTS_FRAME))
		ieee80211_tx_status(entry->ring->net_dev,
			entry->skb, &entry->tx_status);

	rt2x00_set_field32(&word, TXD_W0_VALID, 0);
	rt2x00_desc_write(txd, 0, word);
	CLEAR_FLAG(entry, ENTRY_RTS_FRAME);
	entry->skb = NULL;

	/*
	 * Store the current status of the ring.
	 */
	ring_full = rt2x00_ring_full(entry->ring);

	rt2x00_ring_index_done_inc(entry->ring);

	/*
	 * If the data ring was full before the txdone handler
	 * we must make sure the packet queue in the d80211 stack
	 * is reenabled when the txdone handler has finished.
	 */
	if (ring_full && !rt2x00_ring_full(entry->ring))
		ieee80211_wake_queue(entry->ring->net_dev,
			entry->tx_status.control.queue);
}

static void rt61pci_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(ring->net_dev);
	int index;
	int reg;

	/*
	 * Keep looping while STA_CSR4 contains value data.
	 * at each read the value will be reset to a new value,
	 * which we should check since it contains the ring
	 * and index number of the entry which has been
	 * completely transmitted.
	 */
	while (1) {
		/*
		 * Stop looping when the entry is invalid.
		 */
		rt2x00_register_read(rt2x00dev, STA_CSR4, &reg);
		if (!rt2x00_get_field32(reg, STA_CSR4_VALID))
			break;

		/*
		 * Skip this entry when it contains an invalid
		 * ring identication number.
		 */
		ring = rt2x00_get_ring(rt2x00dev,
			rt2x00_get_field32(reg, STA_CSR4_PID_TYPE));
		if (unlikely(!ring))
			continue;

		/*
		 * Skip this entry when it contains an invalid
		 * index number.
		 */
		index = rt2x00_get_field32(reg, STA_CSR4_PID_SUBTYPE);
		if (unlikely(index >= ring->stats.limit))
			continue;

		/*
		 * Packet received correctly, we can now process it.
		 */
		rt61pci_txdone_entry(&ring->entry[index], reg);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00dev->scan &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_BK]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_BE]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_VI]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_AC_VO]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_PRIO]))
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_READY);
}

static irqreturn_t rt61pci_interrupt(int irq, void *dev_instance,
	struct pt_regs *regs)
{
	struct rt2x00_dev *rt2x00dev = dev_instance;
	u32 reg;

	/*
	 * Get the interrupt sources & saved to local variable.
	 * Write register value back to clear pending interrupts.
	 */
	rt2x00_register_read(rt2x00dev, MCU_INT_SOURCE_CSR, &reg);
	rt2x00_register_write(rt2x00dev, MCU_INT_SOURCE_CSR, reg);

	rt2x00_register_read(rt2x00dev, INT_SOURCE_CSR, &reg);
	rt2x00_register_write(rt2x00dev, INT_SOURCE_CSR, reg);

	if (!reg)
		return IRQ_NONE;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return IRQ_HANDLED;

	/*
	 * Handle interrupts, walk through all bits
	 * and run the tasks, the bits are checked in order of
	 * priority.
	 */

	/*
	 * 1 - Beacon timer expired interrupt.
	 */
	if (rt2x00_get_field32(reg, INT_SOURCE_CSR_BEACON_DONE))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_BEACON].irq_work);

	/*
	 * 2 - Rx ring done interrupt.
	 */
	if (rt2x00_get_field32(reg, INT_SOURCE_CSR_RXDONE))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_RX].irq_work);

	/*
	 * 3 - Tx ring done interrupt.
	 * Don't care about which ring we use to schedule
	 * work to. The handler will run for all TX
	 * related rings except Beacon.
	 */
	if (rt2x00_get_field32(reg, INT_SOURCE_CSR_TXDONE))
		queue_work(rt2x00dev->workqueue,
			&rt2x00dev->ring[RING_AC_BK].irq_work);

	return IRQ_HANDLED;
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt61pci_tx(struct net_device *net_dev, struct sk_buff *skb,
	struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	struct data_entry *entry;
	void *txd;
	struct sk_buff *skb_rts;
	u16 frame_control;
	u32 reg;
	int res;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		dev_kfree_skb_any(skb);
		return NETDEV_TX_OK;
	}

	if (rt2x00_ring_full(ring)) {
		ieee80211_stop_queue(net_dev, control->queue);
		return NETDEV_TX_BUSY;
	}

	/*
	 * If RTS is required. and this frame is not RTS,
	 * create and queue that frame first.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	if (control->use_rts_cts &&
	   (((frame_control & IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_CTL) &&
	    ((frame_control & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_RTS))) {
		skb_rts = rt61pci_create_rts(rt2x00dev,
				ieee80211hdr, control->rts_cts_duration);
		if (!skb_rts) {
			WARNING("Failed to create RTS frame.\n");
			return NETDEV_TX_BUSY;
		}

		res = rt61pci_tx(net_dev, skb_rts, control);
		if (res) {
			WARNING("Failed to send RTS frame.\n");
			return res;
		}
	}

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00pci_desc_addr(entry);
	rt2x00_desc_read(txd, 0, &reg);

	if (rt2x00_get_field32(reg, TXD_W0_OWNER_NIC) ||
	    rt2x00_get_field32(reg, TXD_W0_VALID)) {
		ERROR("Arrived at non-free entry in the non-full queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		ieee80211_stop_queue(net_dev, control->queue);
		return NETDEV_TX_BUSY;
	}

	memcpy(rt2x00pci_data_addr(entry), skb->data, skb->len);
	rt61pci_write_tx_desc(rt2x00dev, txd, skb, control);
	memcpy(&entry->tx_status.control, control, sizeof(*control));
	if (((frame_control & IEEE80211_FCTL_FTYPE) == IEEE80211_FTYPE_CTL) &&
	    ((frame_control & IEEE80211_FCTL_STYPE) == IEEE80211_STYPE_RTS))
		SET_FLAG(entry, ENTRY_RTS_FRAME);
	entry->skb = skb;

	rt2x00_ring_index_inc(ring);

	if (rt2x00_ring_full(ring))
		ieee80211_stop_queue(net_dev, control->queue);

	rt2x00_register_read(rt2x00dev, TX_CNTL_CSR, &reg);
	if (control->queue == IEEE80211_TX_QUEUE_DATA0)
		rt2x00_set_field32(&reg, TX_CNTL_CSR_KICK_TX_AC0, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_DATA1)
		rt2x00_set_field32(&reg, TX_CNTL_CSR_KICK_TX_AC1, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_DATA2)
		rt2x00_set_field32(&reg, TX_CNTL_CSR_KICK_TX_AC2, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_DATA3)
		rt2x00_set_field32(&reg, TX_CNTL_CSR_KICK_TX_AC3, 1);
	else if (control->queue == IEEE80211_TX_QUEUE_DATA4)
		rt2x00_set_field32(&reg, TX_CNTL_CSR_KICK_TX_MGMT, 1);
	rt2x00_register_write(rt2x00dev, TX_CNTL_CSR, reg);

	return NETDEV_TX_OK;
}

static int rt61pci_reset(struct net_device *net_dev)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	rt61pci_disable_radio(rt2x00dev);

	return rt61pci_enable_radio(rt2x00dev);
}

static int rt61pci_add_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	int status;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	/*
	 * Add the new interface.
	 */
	rt2x00_add_interface(&rt2x00dev->interface, conf);

	/*
	 * Initialize interface, and enable the radio when this
	 * is the first interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		/*
		 * Before doing anything else, the MAC address
		 * of this device should be initialized correctly.
		 */
		rt61pci_config_mac_address(rt2x00dev, conf->mac_addr);

		/*
		 * Initialize the device.
		 */
		status = rt61pci_initialize(rt2x00dev);
		if (status)
			return status;

		/*
		 * Only enable radio if the hardware button indicates
		 * the radio is allowed to be enabled.
		 */
		if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO_HW)) {
			status = rt61pci_enable_radio(rt2x00dev);
			if (status)
				return status;
		}

		/*
		 * Enable periodic link tuning if this is a non-monitor
		 * interface. Also set the INTERFACE_INITIALIZED FLAG
		 * to prevent new non-monitor interfaces to be added.
		 */
		if (conf->type != IEEE80211_IF_TYPE_MNTR) {
			queue_delayed_work(rt2x00dev->workqueue,
				&rt2x00dev->link.work, LINK_TUNE_INTERVAL);
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
		} else
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR);
	}

	return 0;
}

static void rt61pci_remove_interface(struct net_device *net_dev,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	/*
	 * Remove the interface.
	 */
	rt2x00_remove_interface(&rt2x00dev->interface, conf);

	/*
	 * When this is a non-monitor mode,
	 * stop the periodic link tuning,
	 * and clear the INTERFACE_INITIALIZED FLAG to allow
	 * new non-monitor interfaces to be added.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR) {
		cancel_rearming_delayed_workqueue(rt2x00dev->workqueue,
			&rt2x00dev->link.work);
		CLEAR_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
	}

	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		rt61pci_disable_radio(rt2x00dev);

	/*
	 * Check if we still have 1 non-monitor or a monitor
	 * interface enabled. In that case we should update the
	 * registers.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			rt61pci_config_type(rt2x00dev,
				rt2x00dev->interface.type);
		else
			rt61pci_config_type(rt2x00dev,
				IEEE80211_IF_TYPE_MNTR);
	}
}

static int rt61pci_config(struct net_device *net_dev,
	struct ieee80211_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	u32 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt61pci_disable_radio(rt2x00dev);
		else {
			rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
			rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
		}
	}

	rt61pci_config_channel(rt2x00dev,
		conf->channel_val, conf->channel, conf->freq,
		conf->power_level);
	rt61pci_config_txpower(rt2x00dev, conf->power_level);
	rt61pci_config_antenna(rt2x00dev, conf->antenna_sel, conf->phymode);
	rt61pci_config_duration(rt2x00dev, conf->short_slot_time);
	rt61pci_config_phymode(rt2x00dev, conf->phymode);

	/*
	 * Reenable RX only if the radio should be on.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
		rt2x00_set_field32(&reg, TXRX_CSR0_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);
	} else if (conf->radio_enabled)
		return rt61pci_enable_radio(rt2x00dev);

	return 0;
}

static int rt61pci_config_interface(struct net_device *net_dev, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	/*
	 * Monitor mode does not need configuring.
	 * If the given type does not match the configured type,
	 * there has been a problem.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00dev->interface.type)
		return -EINVAL;

	/*
	 * If the interface does not work in master mode,
	 * then the bssid value in the interface structure
	 * should now be set.
	 */
	if (conf->type != IEEE80211_IF_TYPE_AP)
		memcpy(&rt2x00dev->interface.bssid, conf->bssid, ETH_ALEN);

	/*
	 * Enable configuration.
	 */
	rt61pci_config_type(rt2x00dev, conf->type);
	rt61pci_config_bssid(rt2x00dev, &rt2x00dev->interface.bssid[0]);

	return 0;
}

static void rt61pci_set_multicast_list(struct net_device *net_dev,
	unsigned short flags, int mc_count)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	int update = 0;

	if (GET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC)) {
			rt2x00dev->interface.promisc = 0;
			update = 1;
		}
	} else {
		if (flags & IFF_PROMISC) {
			rt2x00dev->interface.promisc = 1;
			update = 1;
		}
	}

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (update && !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt61pci_config_promisc(rt2x00dev,
			rt2x00dev->interface.promisc);
}

static void rt61pci_scan(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;

	if (unlikely(!rt2x00dev->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00_wait_scan(rt2x00dev->scan))
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00dev->scan->state == IEEE80211_SCAN_START) {
		rt61pci_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.scan_phymode);

		rt61pci_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.scan_channel_val,
			rt2x00dev->scan->conf.scan_channel,
			rt2x00dev->scan->conf.scan_freq,
			rt2x00dev->scan->conf.scan_power_level);
	} else {
		rt61pci_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.running_phymode);

		rt61pci_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.running_channel_val,
			rt2x00dev->scan->conf.running_channel,
			rt2x00dev->scan->conf.running_freq,
			rt2x00dev->scan->conf.scan_power_level);
	}

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;
}

static int rt61pci_passive_scan(struct net_device *net_dev,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00dev->scan)
		return -EBUSY;

	/*
	 * Check if the radio is enabled.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00dev->scan = kmalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00dev->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt61pci_tx(net_dev, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	rt2x00_start_scan(rt2x00dev->scan, conf, state);

	/*
	 * Queue work.
	 */
	INIT_WORK(&rt2x00dev->scan->work, rt61pci_scan, rt2x00dev);
	if (!queue_work(rt2x00dev->workqueue, &rt2x00dev->scan->work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;

	return -EIO;
}

static int rt61pci_get_stats(struct net_device *net_dev,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	/*
	 * The dot11ACKFailureCount is updated in interrupt time.
	 * TODO: dot11FCSErrorCount, dot11RTSFailureCount and
	 * dot11RTSSuccessCount are never updated,
	 * we need to find a method to see where we can update
	 * those statistics from.
	 */
	memcpy(stats, &rt2x00dev->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt61pci_set_retry_limit(struct net_device *net_dev,
	u32 short_retry, u32 long_retry)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR4, &reg);
	rt2x00_set_field32(&reg, TXRX_CSR4_LONG_RETRY_LIMIT, long_retry);
	rt2x00_set_field32(&reg, TXRX_CSR4_SHORT_RETRY_LIMIT, short_retry);
	rt2x00_register_write(rt2x00dev, TXRX_CSR4, reg);

	return 0;
}

static int rt61pci_conf_tx(struct net_device *net_dev,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	struct data_ring *ring;

	ring = rt2x00_get_ring(rt2x00dev, queue);
	if (unlikely(!ring))
		return -EINVAL;

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT61 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 4; /* cw_min: 2^4 = 16. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	if (!params->aifs)
		ring->tx_params.aifs = 2;

	INFO("Configured TX ring %d - CWmin: %d, CWmax: %d, Aifs: %d.\n",
		queue, ring->tx_params.cw_min, ring->tx_params.cw_max,
		ring->tx_params.aifs);

	return 0;
}

static int rt61pci_get_tx_stats(struct net_device *net_dev,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00dev->ring[RING_AC_BK].stats,
		sizeof(rt2x00dev->ring[RING_AC_BK].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00dev->ring[RING_AC_BE].stats,
		sizeof(rt2x00dev->ring[RING_AC_BE].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA2],
		&rt2x00dev->ring[RING_AC_VI].stats,
		sizeof(rt2x00dev->ring[RING_AC_VI].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA3],
		&rt2x00dev->ring[RING_AC_VO].stats,
		sizeof(rt2x00dev->ring[RING_AC_VO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA4],
		&rt2x00dev->ring[RING_PRIO].stats,
		sizeof(rt2x00dev->ring[RING_PRIO].stats));

	return 0;
}

static u64 rt61pci_get_tsf(struct net_device *net_dev)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	u64 tsf;
	u32 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR13, &reg);
	tsf = (u64)rt2x00_get_field32(reg, TXRX_CSR13_HIGH_TSFTIMER) << 32;
	rt2x00_register_read(rt2x00dev, TXRX_CSR12, &reg);
	tsf |= rt2x00_get_field32(reg, TXRX_CSR12_LOW_TSFTIMER);

	return tsf;
}

static void rt61pci_reset_tsf(struct net_device *net_dev)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	u32 reg = 0;

	rt2x00_register_write(rt2x00dev, TXRX_CSR12, reg);
	rt2x00_register_write(rt2x00dev, TXRX_CSR13, reg);
}

static int rt61pci_beacon_update(struct net_device *net_dev,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00dev->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	memcpy(rt2x00pci_data_addr(entry), skb->data, skb->len);
	rt61pci_write_tx_desc(rt2x00dev, rt2x00pci_desc_addr(entry),
		skb, control);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt61pci_alloc_rings(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	rt2x00dev->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00dev->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	for (i = 0; i < RING_NUM; i++) {
		/*
		 *Set device structure.
		 */
		rt2x00dev->ring[i].net_dev =
			pci_get_drvdata(rt2x00dev_pci(rt2x00dev));

		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00dev->ring[i].tx_params.aifs = 2;
		rt2x00dev->ring[i].tx_params.cw_min = 5;
		rt2x00dev->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt61pci_init_eeprom(struct rt2x00_dev *rt2x00dev)
{
	struct ieee80211_conf *conf = ieee80211_get_hw_conf(
		pci_get_drvdata(rt2x00dev_pci(rt2x00dev)));
	u32 reg;
	u16 value;
	u16 eeprom;
	u16 device;

	/*
	 * 1 - Detect EEPROM width.
	 */
	rt2x00_register_read(rt2x00dev, E2PROM_CSR, &reg);
	if (rt2x00_get_field32(reg, E2PROM_CSR_TYPE_93C46))
		rt2x00dev->eeprom_width = EEPROM_WIDTH_93C46;
	else
		rt2x00dev->eeprom_width = EEPROM_WIDTH_93C66;

	/*
	 * 2 - Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_ANTENNA, &eeprom);

	/*
	 * 3 - Identify RF chipset.
	 * To determine the RT chip we have to read the
	 * PCI header of the device.
	 */
	pci_read_config_word(rt2x00dev_pci(rt2x00dev),
		PCI_CONFIG_HEADER_DEVICE, &device);
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
	rt2x00_set_chip(&rt2x00dev->chip, device, value, reg);

	if (!rt2x00_rf(&rt2x00dev->chip, RF5225) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF5325) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2527) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2529))
		return -ENODEV;

	/*
	 * 4 - Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	conf->antenna_sel = rt2x00_get_field16(eeprom,
				EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * 5 - Read the Frame type.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_FRAME_TYPE))
		SET_FLAG(rt2x00dev, CONFIG_FRAME_TYPE);

	/*
	 * 6 - Determine number of antenna's.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_NUM) == 2)
		SET_FLAG(rt2x00dev, CONFIG_DOUBLE_ANTENNA);

	/*
	 * Detect if this device has an hardware controlled radio.
	 */
	if (rt2x00_get_field16(eeprom, EEPROM_ANTENNA_HARDWARE_RADIO))
		SET_FLAG(rt2x00dev, DEVICE_SUPPORT_HW_BUTTON);

	/*
	 * 7 - Read frequency offset and RF programming sequence.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_FREQ, &eeprom);
	if (rt2x00_get_field16(eeprom, EEPROM_FREQ_SEQ_MASK) != 0xff &&
	    rt2x00_get_field16(eeprom, EEPROM_FREQ_SEQ))
		SET_FLAG(rt2x00dev, CONFIG_RF_SEQUENCE);

	rt2x00dev->freq_offset = rt2x00_get_field16(eeprom,
					EEPROM_FREQ_OFFSET);
	if (rt2x00dev->freq_offset == 0xff)
		rt2x00dev->freq_offset = 0;

	/*
	 * 8 - Read external LNA informations.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_EXTERNAL_LNA_A))
		SET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA_A);
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_EXTERNAL_LNA_BG))
		SET_FLAG(rt2x00dev, CONFIG_EXTERNAL_LNA_BG);

	/*
	 * 9 - Store led settings, for correct led behaviour.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_LED, &eeprom);

	/*
	 * If the eeprom value is invalid,
	 * switch to default led mode.
	 */
	if (eeprom == 0xffff)
		rt2x00dev->led_mode = LED_MODE_DEFAULT;
	else
		rt2x00dev->led_mode = rt2x00_get_field16(eeprom,
			EEPROM_LED_LED_MODE);

	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_LED_MODE,
		rt2x00dev->led_mode);
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_0,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_0));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_1,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_1));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_2,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_2));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_3,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_3));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_GPIO_4,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_GPIO_4));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_ACT,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_ACT));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_READY_BG,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_G));
	rt2x00_set_field16(&rt2x00dev->led_reg, MCU_LEDCS_POLARITY_READY_A,
		rt2x00_get_field16(eeprom, EEPROM_LED_POLARITY_RDY_A));

	/*
	 * 11 - Read BBP data from EEPROM and store in private structure.
	 */
	rt2x00dev->eeprom = kzalloc(EEPROM_BBP_SIZE * sizeof(u16), GFP_KERNEL);
	if (!rt2x00dev->eeprom)
		return -ENOMEM;

	rt2x00_eeprom_multiread(rt2x00dev, EEPROM_BBP_START,
		rt2x00dev->eeprom, EEPROM_BBP_SIZE);

	return 0;
}

static int rt61pci_init_mac(struct rt2x00_dev *rt2x00dev)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00dev_pci(rt2x00dev));

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED_MAC))
		return 0;

	memset(&net_dev->perm_addr[0], 0, sizeof(net_dev->perm_addr));

	/*
	 * Read MAC address from EEPROM.
	 */
	rt2x00_eeprom_multiread(rt2x00dev, EEPROM_MAC_ADDR_0,
		(u16*)&net_dev->perm_addr[0], 3);

	/*
	 * Check if a valid MAC address has been read.
	 */
	if (!is_valid_ether_addr(&net_dev->perm_addr[0]))
		return -EINVAL;

	/*
	 * Copy to perm_addr to dev_addr structure.
	 */
	memcpy(&net_dev->dev_addr[0], &net_dev->perm_addr[0],
		sizeof(net_dev->perm_addr));
	net_dev->addr_len = 6;

	/*
	 * Write MAC address to register.
	 */
	rt61pci_config_mac_address(rt2x00dev, &net_dev->perm_addr[0]);

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED_MAC);
	return 0;
}

static int rt61pci_init_hw(struct rt2x00_dev *rt2x00dev)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00dev_pci(rt2x00dev));
	struct ieee80211_hw *hw = &rt2x00dev->hw;

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW))
		return 0;

	/*
	 * IEEE80211 Function callbacks.
	 */
	hw->tx			= rt61pci_tx;
	hw->reset		= rt61pci_reset;
	hw->add_interface	= rt61pci_add_interface;
	hw->remove_interface	= rt61pci_remove_interface;
	hw->config		= rt61pci_config;
	hw->config_interface	= rt61pci_config_interface;
	hw->set_multicast_list	= rt61pci_set_multicast_list;
	hw->passive_scan	= rt61pci_passive_scan;
	hw->get_stats		= rt61pci_get_stats;
	hw->set_retry_limit	= rt61pci_set_retry_limit;
	hw->conf_tx		= rt61pci_conf_tx;
	hw->get_tx_stats	= rt61pci_get_tx_stats;
	hw->get_tsf		= rt61pci_get_tsf;
	hw->reset_tsf		= rt61pci_reset_tsf;
	hw->beacon_update	= rt61pci_beacon_update;

	/*
	 * IEEE80211 Variables.
	 */
	hw->version = IEEE80211_VERSION;
	hw->name = DRV_NAME;
	hw->host_gen_beacon = 1;
	hw->host_gen_beacon_template = 0;
	hw->device_hides_wep = 0;
	hw->rx_includes_fcs = 0;
	hw->host_broadcast_ps_buffering = 1;
	hw->wep_include_iv = 1;
	hw->data_nullfunc_ack = 1;
	hw->no_tkip_wmm_hwaccel = 1;
	hw->extra_hdr_room = 0;
	hw->device_strips_mic = 0;
	hw->monitor_during_oper = 1;
	hw->fraglist = 0;

	/*
	 * We have full WMM capabilities.
	 */
	hw->queues = RING_NUM_TX;

	if (ieee80211_register_hw(net_dev, hw))
		return -EIO;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt61pci_init_hw_channels(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u16 eeprom;

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS |
			IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS |
		IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
		for (i = 14; i < 38; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS |
				IEEE80211_CHAN_W_ACTIVE_SCAN |
				IEEE80211_CHAN_W_SCAN;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	if (!GET_FLAG(rt2x00dev, CONFIG_RF_SEQUENCE)) {
		static const u32 vals[] = {
			0x00004786, 0x00004786, 0x0000478a, 0x0000478a,
			0x0000478e, 0x0000478e, 0x00004792, 0x00004792,
			0x00004796, 0x00004796, 0x0000479a, 0x0000479a,
			0x0000479e, 0x000047a2
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];

		if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
		    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
			static const u32 vals[] = {
				0x0000499a, 0x000049a2, 0x000049a6, 0x000049aa,
				0x000049ae, 0x000049b2, 0x000049ba, 0x000049be,
				0x00004a2a, 0x00004a2e, 0x00004a32, 0x00004a36,
				0x00004a3a, 0x00004a82, 0x00004a86, 0x00004a8a,
				0x00004a8e, 0x00004a92, 0x00004a9a, 0x00004aa2,
				0x00004aa6, 0x00004aae, 0x00004ab2, 0x00004ab6
			};

			struct ieee80211_channel *chan = channels + 14;

			for (i = 0; i < ARRAY_SIZE(vals); i++)
				(chan++)->val = vals[i];
		}
	} else {
		static const u32 vals[] = {
			0x00004786, 0x00004786, 0x0000478a, 0x0000478a,
			0x0000478e, 0x0000478e, 0x00004792, 0x00004792,
			0x00004796, 0x00004796, 0x0000479a, 0x0000479a,
			0x0000479e, 0x000047a2
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];

		if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
		    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
			static const u32 vals[] = {
				0x0004481a, 0x00044682, 0x00044686, 0x0004468e,
				0x00044692, 0x0004469a, 0x000446a2, 0x000446a6,
				0x0004489a, 0x000448a2, 0x000448aa, 0x000448b2,
				0x000448ba, 0x00044702, 0x00044706, 0x0004470e,
				0x00044712, 0x0004471a, 0x00044722, 0x0004472e,
				0x00044736, 0x0004490a, 0x00044912, 0x0004491a
			};

			struct ieee80211_channel *chan = channels + 14;

			for (i = 0; i < ARRAY_SIZE(vals); i++)
				(chan++)->val = vals[i];
		}
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_G_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev,
			EEPROM_TXPOWER_G_START + i, &eeprom);

		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_1));

		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_G_2));
	}

	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
		for (; i < EEPROM_TXPOWER_A_SIZE; i++) {
			rt2x00_eeprom_read(rt2x00dev,
				EEPROM_TXPOWER_A_START + i, &eeprom);

			channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
				rt2x00_get_field16(eeprom,
					EEPROM_TXPOWER_A_1));

			channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
				rt2x00_get_field16(eeprom,
					EEPROM_TXPOWER_A_2));
		}
	}
}

static void rt61pci_init_hw_rates(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	device_rate_entry(&rates[0], 10, 0x001, 0x00, IEEE80211_RATE_CCK);
	device_rate_entry(&rates[1], 20, 0x003, 0x01, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[2], 55, 0x007, 0x02, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[3], 110, 0x00f, 0x03, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[4], 60, 0x01f, 0x0b, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[5], 90, 0x03f, 0x0f, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[6], 120, 0x07f, 0x0a, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[7], 180, 0x0ff, 0x0e, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[8], 240, 0x1ff, 0x09, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[9], 360, 0x3ff, 0x0d, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[10], 480, 0x7ff, 0x08, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[11], 540, 0xfff, 0x0c, IEEE80211_RATE_OFDM);
}

static int rt61pci_init_hw_modes(struct rt2x00_dev *rt2x00dev)
{
	struct net_device *net_dev = pci_get_drvdata(rt2x00dev_pci(rt2x00dev));
	struct ieee80211_hw *hw = &rt2x00dev->hw;
	int num_modes;
	int num_channels;

	/*
	 * RF2527 and RF2529 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5225 and RF5325 also supports 802.11a, so allocate an
	 * additional 23 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
		num_modes = 3;
		num_channels = 38;
	}

	hw->num_modes = num_modes;
	hw->modes =
		kzalloc((sizeof(struct ieee80211_hw_modes) * num_modes),
		GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->channels =
		kzalloc((sizeof(struct ieee80211_channel) * num_channels),
		GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->rates =
		kzalloc((sizeof(struct ieee80211_rate) * 12),
		GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	hw->modes[0].mode = MODE_IEEE80211G;
	hw->modes[0].num_channels = 14;
	hw->modes[0].num_rates = 12;

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	hw->modes[1].mode = MODE_IEEE80211B;
	hw->modes[1].num_channels = 14;
	hw->modes[1].num_rates = 4;
	hw->modes[1].channels = hw->modes[0].channels;
	hw->modes[1].rates = hw->modes[0].rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF5225) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5325)) {
		hw->modes[2].mode = MODE_IEEE80211A;
		hw->modes[2].num_channels = 38;
		hw->modes[2].num_rates = 8;
		hw->modes[2].channels = &hw->modes[0].channels[14];
		hw->modes[2].rates = &hw->modes[0].rates[4];
	}

	rt61pci_init_hw_channels(rt2x00dev, hw->modes[0].channels);
	rt61pci_init_hw_rates(rt2x00dev, hw->modes[0].rates);

	/*
	 * xr_end is only used on Atheros cards.
	 */
	hw->modes->xr_end = 0;

	return ieee80211_update_hw(net_dev, hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static void rt61pci_free_dev(struct net_device *net_dev)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	/*
	 * Shutdown poll_timer for hardware button.
	 */
	rt61pci_button_stop(rt2x00dev);

	/*
	 * Free ring structures.
	 */
	kfree(rt2x00dev->ring);
	rt2x00dev->ring = NULL;

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00dev->eeprom);

	/*
	 * Release CSR memory.
	 */
	if (likely(rt2x00dev->csr_addr)) {
		iounmap(rt2x00dev->csr_addr);
		rt2x00dev->csr_addr = NULL;
	}

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00dev->workqueue)) {
		destroy_workqueue(rt2x00dev->workqueue);
		rt2x00dev->workqueue = NULL;
	}

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00dev->hw.modes)) {
		kfree(rt2x00dev->hw.modes->channels);
		kfree(rt2x00dev->hw.modes->rates);
		kfree(rt2x00dev->hw.modes);
		rt2x00dev->hw.modes = NULL;
	}
}

static int rt61pci_alloc_dev(struct pci_dev *pci_dev,
	struct net_device *net_dev)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	rt2x00dev->dev = pci_dev;

	rt2x00dev->csr_addr = ioremap(
		pci_resource_start(rt2x00dev_pci(rt2x00dev), 0),
		pci_resource_len(rt2x00dev_pci(rt2x00dev), 0));
	if (!rt2x00dev->csr_addr) {
		ERROR("Ioremap failed.\n");
		return -ENOMEM;
	}

	rt2x00dev->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00dev->workqueue)
		goto exit;

	/*
	 * Initialize configuration work.
	 */
	INIT_WORK(&rt2x00dev->link.work, rt61pci_link_tuner, rt2x00dev);

	/*
	 * Reset current working type.
	 */
	rt2x00dev->interface.type = -EINVAL;

	/*
	 * Initialize scanning attributes.
	 */
	rt2x00dev->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt61pci_alloc_rings(rt2x00dev))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt61pci_init_eeprom(rt2x00dev) ||
	    rt61pci_init_mac(rt2x00dev) ||
	    rt61pci_init_hw(rt2x00dev) ||
	    rt61pci_init_hw_modes(rt2x00dev)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	if (rt61pci_init_firmware(rt2x00dev)) {
		ERROR("Failed to load Firmware.\n");
		goto exit;
	}

	/*
	 * If required start hardware button polling.
	 */
	rt61pci_button_start(rt2x00dev);

	return 0;

exit:
	rt61pci_free_dev(net_dev);

	return -ENODEV;
}

/*
 * PCI driver handlers.
 */
static int rt61pci_probe(struct pci_dev *pci_dev,
	const struct pci_device_id *id)
{
	struct net_device *net_dev;
	int status;

	status = pci_request_regions(pci_dev, pci_name(pci_dev));
	if (status) {
		ERROR("PCI request regions failed.\n");
		return status;
	}

	status = pci_enable_device(pci_dev);
	if (status) {
		ERROR("Enable device failed.\n");
		goto exit_release_regions;
	}

	pci_set_master(pci_dev);

	if (pci_set_mwi(pci_dev))
		NOTICE("MWI not available.\n");

	if (pci_set_dma_mask(pci_dev, DMA_64BIT_MASK) &&
	    pci_set_dma_mask(pci_dev, DMA_32BIT_MASK)) {
		ERROR("PCI DMA not supported.\n");
		status = -EIO;
		goto exit_disable_device;
	}

	net_dev = ieee80211_alloc_hw(sizeof(struct rt2x00_dev), NULL);
	if (!net_dev) {
		ERROR("Failed to allocate hardware.\n");
		status = -ENOMEM;
		goto exit_disable_device;
	}

	SET_ETHTOOL_OPS(net_dev, &rt61pci_ethtool_ops);
	SET_NETDEV_DEV(net_dev, &pci_dev->dev);

	pci_set_drvdata(pci_dev, net_dev);

	status = rt61pci_alloc_dev(pci_dev, net_dev);
	if (status) {
		ERROR("Failed to allocate device.\n");
		goto exit_free_device;
	}

	ieee80211_netif_oper(net_dev, NETIF_ATTACH);

	return 0;

exit_free_device:
	ieee80211_free_hw(net_dev);

exit_disable_device:
	if (status != -EBUSY)
		pci_disable_device(pci_dev);

exit_release_regions:
	pci_release_regions(pci_dev);

	pci_set_drvdata(pci_dev, NULL);

	return status;
}

static void rt61pci_remove(struct pci_dev *pci_dev)
{
	struct net_device *net_dev = pci_get_drvdata(pci_dev);
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);

	rt61pci_disable_radio(rt2x00dev);

	ieee80211_netif_oper(net_dev, NETIF_DETACH);

	rt61pci_uninitialize(rt2x00dev);

	rt61pci_free_dev(net_dev);

	ieee80211_unregister_hw(net_dev);

	ieee80211_free_hw(net_dev);

	pci_set_drvdata(pci_dev, NULL);

	pci_disable_device(pci_dev);

	pci_release_regions(pci_dev);
}

#ifdef CONFIG_PM
static int rt61pci_suspend(struct pci_dev *pci_dev, pm_message_t state)
{
	struct net_device *net_dev = pci_get_drvdata(pci_dev);
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	int status;

	NOTICE("Going to sleep.\n");

	/*
	 * Disable the radio.
	 */
	rt61pci_disable_radio(rt2x00dev);
	ieee80211_netif_oper(net_dev, NETIF_DETACH);

	/*
	 * Set device mode to sleep for power management.
	 */
	status = rt61pci_set_state(rt2x00dev, STATE_SLEEP);
	if (status)
		return status;

	/*
	 * Uninitialize device.
	 */
	rt61pci_uninitialize(rt2x00dev);

	/*
	 * Uninitialize hardware.
	 */
	rt61pci_free_dev(net_dev);

	/*
	 * Disable PCI.
	 */
	pci_save_state(pci_dev);
	pci_disable_device(pci_dev);
	return pci_set_power_state(pci_dev, pci_choose_state(pci_dev, state));
}

static int rt61pci_resume(struct pci_dev *pci_dev)
{
	struct net_device *net_dev = pci_get_drvdata(pci_dev);
	struct rt2x00_dev *rt2x00dev = ieee80211_dev_hw_data(net_dev);
	int status;

	NOTICE("Waking up.\n");

	/*
	 * Enable PCI.
	 */
	if (pci_set_power_state(pci_dev, PCI_D0) ||
	    pci_enable_device(pci_dev) ||
	    pci_restore_state(pci_dev)) {
		ERROR("Failed to resume device.\n");
		return -EIO;
	}

	/*
	 * Initialize hardware.
	 */
	status = rt61pci_alloc_dev(pci_dev, net_dev);
	if (status) {
		ERROR("Failed to allocate device.\n");
		return status;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	status = rt61pci_set_state(rt2x00dev, STATE_AWAKE);
	if (status)
		return status;

	ieee80211_netif_oper(net_dev, NETIF_ATTACH);

	return 0;
}
#endif /* CONFIG_PM */

/*
 * RT61pci module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct pci_device_id rt61pci_device_table[] = {
	/* RT2561s */
	{ PCI_DEVICE(0x1814, 0x0301) },
	/* RT2561 v2 */
	{ PCI_DEVICE(0x1814, 0x0302) },
	/* RT2661 */
	{ PCI_DEVICE(0x1814, 0x0401) },
	{ 0, }
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT61 PCI & PCMCIA Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2561, RT2561s & RT2661 "
	"PCI & PCMCIA chipset based cards");
MODULE_DEVICE_TABLE(pci, rt61pci_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT61PCI_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT61PCI_DEBUG */

static struct pci_driver rt61pci_driver = {
	.name		= DRV_NAME,
	.id_table	= rt61pci_device_table,
	.probe		= rt61pci_probe,
	.remove		= __devexit_p(rt61pci_remove),
#ifdef CONFIG_PM
	.suspend	= rt61pci_suspend,
	.resume		= rt61pci_resume,
#endif /* CONFIG_PM */
};

static int __init rt61pci_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return pci_register_driver(&rt61pci_driver);
}

static void __exit rt61pci_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	pci_unregister_driver(&rt61pci_driver);
}

module_init(rt61pci_init);
module_exit(rt61pci_exit);
