/*
	Copyright (C) 2006 Ivo van Doorn

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	RF button support
	Laptops are quite often equiped with a RF button to enable or
	disable the radio of the network device attached to that button.
	This network device usually is an integrated wireless network device,
	or bluetooth device.
	Some of these devices will disable the radio automaticly when the
	RF button has been pressed, while other devices need to be polled
	for the RF button status.
	But in all cases the only interface that will have its radio disabled
	will be the device that has the RF button attached to it. But it could
	be desired that userspace performs this disabling of all radios in case
	there are also interfaces without a RF button that need to be disabled.

	The rfkill driver will contain a list of all devices with a RF button,
	hardware drivers need to register their hardware to teh rfkill
	interface it can take care of everything. If the RF button requires
	polling to obtain the status this will be handled by rfkill as well.
	Once the status of the button has changed and the hardware does not
	automaticly enable or disable the radio rfkill provides with the
	interface to do this correctly.

	For each registered hardware button an input device will be created.
	If this input device has been opened by the user, rfkill will send a
	signal to userspace instead of the hardware about the new button
	status. This will allow userpace to perform the correct steps
	in order to bring down all interfaces.
 */

#ifndef RFKILL_H
#define RFKILL_H

#include <linux/list.h>
#include <linux/input.h>

#define RFKILL_POLL_DELAY	( HZ / 10 ) /* 100 ms */

/**
 * struct rfkill - rfkill button control structure.
 * @dev_name: Name of the interface. This will become the name
 * 	of the input device which will be created for this button.
 * @data: Private data which will be passed along with the radio and polling
 * 	handlers.
 * @poll(unsigned long data): Optional handler which will frequently be
 * 	called to determine the current status of the RF button.
 * @enable_radio(unsigned long data): Optional handler to enable the radio
 * 	once the RF button has been pressed and the hardware does enable
 * 	the radio automaticly.
 * @disable_radio(unsigned long data): Optional handler to disable the radio
 * 	once the RF button has been pressed and the hardware does disable
 * 	the radio automaticly.
 * @current_status: Contains the current status of the radio as it was
 * 	previously indicated by the radio. This field may only be changed
 * 	by the driver at initialization time.
 */
struct rfkill {
	const char *dev_name;

	unsigned long data;

	int (*poll)(unsigned long data);
	void (*enable_radio)(unsigned long data);
	void (*disable_radio)(unsigned long data);

	unsigned int current_status;

	/*
	 * These fields are private to rfkill, and
	 * should not be used by the RF button driver.
	 */
	struct list_head entry;
	struct input_dev *input_dev;
};

void rfkill_button_event(struct rfkill *rfkill, int status);
int rfkill_add_device(struct rfkill *rfkill);
void rfkill_del_device(struct rfkill *rfkill);

#endif /* RFKILL_H */
