/* planactions.c - `apply' and `retry' commands for interactive qmv.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* C89 */
#include <stdlib.h>
/* POSIX/gnulib */
#include <stdbool.h>
/* Gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* common */
#include "common/error.h"
#include "common/memory.h"
/* qmv */
#include "qmv.h"

void
apply_command(char **args)
{
	apply_plan(plan);
}

void
retry_command(char **args)
{
	retry_plan(plan);
}

bool
apply_plan(RenamePlan *plan)
{
	Iterator *it;
	bool changed = false;
	bool failed = false;

	if (plan == NULL) {
		printf(_("no plan - use `list' and `edit'\n"));
		return false;
	}
	if (!llist_is_empty(plan->error)) {
		printf(_("plan contains errors - will only rename OK files\n"));
		return false;
	}
	if (llist_is_empty(plan->ok)) {
		printf(_("plan is empty (no changes made).\n"));
		return true;
	}

	if (!cwd_to_renames_directory())
		return false;

	for (it = llist_iterator(plan->ok); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		char *name_old;
		char *name_new;
		
		if (spec->renamed == RENAME_FAILED)
			failed = true;
		if (spec->renamed != RENAME_UNDONE)
			continue;
		
		name_old = quote_output_file(spec->old_name);
		name_new = quote_output_file(spec->new_name);
		printf(_("%s -> %s\n"), name_old, name_new);
		free(name_old);
		free(name_new);
		
		changed = true;
		if (!dummy) {
			if (rename(spec->old_name, spec->new_name) < 0) {
				warn_errno(_("cannot rename %s to %s"), spec->old_name, spec->new_name);
				spec->renamed = RENAME_FAILED;
				failed = true;
				printf(_("rename failed. `apply' aborted.\n"));
				break;
			} else {
				spec->renamed = RENAME_COMPLETE;
			}
		}
	}
	iterator_free(it);

	if (!changed)
		printf(_("nothing to apply - no changes made.\n"));
	if (failed)
		printf(_("there are failed renames. retry them with `retry'.\n"));

	if (!cwd_from_renames_directory())
		return false;

	return !failed;
}

bool
retry_plan(RenamePlan *plan)
{
	Iterator *it;
	bool changed = false;

	if (plan == NULL) {
		printf(_("no plan - use `list' and `edit'\n"));
		return false;
	}
	if (llist_is_empty(plan->ok)) {
		printf(_("plan is empty (no changes made).\n"));
		return true;
	}

	for (it = llist_iterator(plan->ok); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->renamed != RENAME_FAILED)
			continue;
		
		printf(_("%s -> %s\n"), spec->old_name, spec->new_name);
		changed = true;
		if (!dummy) {
			if (rename(spec->old_name, spec->new_name) < 0) {
				warn_errno(_("cannot rename %s to %s"), spec->old_name, spec->new_name);
				spec->renamed = RENAME_FAILED;
				printf(_("renaming failed. `retry' aborted.\n"));
				break;
			} else {
				spec->renamed = RENAME_COMPLETE;
			}
		}
		changed = true;
	}
	iterator_free(it);

	if (!changed)
		printf(_("no failed renames - no changes made.\n"));

	return true;
}
