/******************************************************************************
*		       							      *
* console.c (part of rCalc)					       	      *
* Copyright 2000 Gary Benson <rat@spunge.org>			       	      *
*								       	      *
* This program is free software; you can redistribute it and/or modify 	      *
* it under the terms of the GNU General Public License as published by 	      *
* the Free Software Foundation; either version 2 of the License, or    	      *
* (at your option) any later version.				       	      *
*								       	      *
* This program is distributed in the hope that it will be useful,      	      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       	      *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	       	      *
* GNU General Public License for more details.			       	      *
*								       	      *
* You should have received a copy of the GNU General Public License    	      *
* along with this program; if not, write to the Free Software	       	      *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.	       	      *
*								       	      *
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <readline/readline.h>
#include <readline/history.h>
#include <gnome.h>

#include "console.h"
#include "config.h"
#include "engine.h"

/* Command history specifications.
*/
#define HISTORY_FILE		".rcalc_history"
#define HISTORY_MAXLINES	100

/* Various different colours.
*/
#define DEFAULTCOLOUR	"\x1B[0;40;37m"
#define PROMPTCOLOUR	"\x1B[0;40;34m"
#define ERRORCOLOUR	"\x1B[0;40;31m"
#define DEBUGCOLOUR	"\x1B[0;40;32m"

/* The prompt to use.
*/
static char user_prompt[256];

/* Print debugging messages?
*/
/* #define PRINT_DEBUGGING_MESSAGES */

/*****************************************************************************/

/* Test a completion for acceptability.
*/
static int completionAcceptable( char *c )
{
	return 0;
}

/* Generate one completion.
*/
static char *completionGenerator( char *text, int state )
{
	return NULL;
}

/* Ignore some completions - basically remove all the filenames that
** readline has put there.
*/
static void ignoreCompletions( char **names )
{
	char **temp;
	int num, i;

	/* Handle only one completion - if so, the prefix is
	** not supplied as names[0] - names[0] is the completion.
	*/
	if( names[1]==NULL )
	{
		if( !completionAcceptable( names[0] ) )
		{
			free( names[0] );
			names[0] = NULL;
		}
		return;
	}
	
	/* Handle multiple completions
	*/
	for( num=-1, temp=names; *temp; temp++, num++ );
	for( i=1; i<=num; i++ )
	{
		if( !completionAcceptable( names[i] ) )
		{
			int j;
			
			free( names[i] );
			for( j=i; j<=num; j++ )
			{
				names[j] = names[j+1];
			}
			i--;
			num--;
		}
	}

	/* If none are acceptable, free the prefix.
	*/
	if( num==0 )
	{
		free( names[0] );
		names[0] = NULL;
	}
	
	/* If one is acceptable, free the prefix and replace
	** with the completion.
	*/
	if( num==1 )
	{
		free( names[0] );
		names[0] = names[1];
		names[1] = NULL;
	}
}

/* Custom completion function.
*/
static char **tabComplete( char *text, int start, int end )
{
	return completion_matches( text, completionGenerator );
}

/*****************************************************************************/

/* Strip whitespace from the start and end of a string
*/
static char *stripLeadAndTrailSpace( char *string )
{
	char *s, *t;

	for( s = string; isspace(*s); s++ );
    
	if( *s == 0 ) return(s);

	t = s + strlen (s) - 1;
	while( t>s && isspace(*t) ) t--;
	*++t = '\0';

	return s;
}

/*****************************************************************************/

static char *historyFile;	/* Filename of the command history	     */

void rCalc_console_Startup( void )
{
	char *home;

	/* Build the filename for the history file. If it isn't
	** successful then the history won't be loaded or saved.
	*/
	if(( home = getenv( "HOME" ) ))
	{
		if(( historyFile = malloc( strlen(home) +
					  strlen(HISTORY_FILE) +2 ) ))
		{
			sprintf( historyFile, "%s/%s",
				 home, HISTORY_FILE );
		}
	}
	else historyFile = NULL;

	/* Initialise readline - set our name to allow conditional
	** parsing of the ~/.inputrc file and hook in our completion
	** function.
	*/
	rl_readline_name = PACKAGE;
	rl_attempted_completion_function = tabComplete;
	rl_ignore_some_completions_function = (Function *)ignoreCompletions;

	/* Initialise the readline history - read in the previous history
	** file and then set the maximum number of lines to remember.
	*/
	if( historyFile )
	{
		read_history( historyFile );
		stifle_history( HISTORY_MAXLINES );
	}

	/* Create the prompt
	*/
	sprintf( user_prompt, "%c%c%s%c%c%s%c%c%s%c%c ",
		 '\001', RL_PROMPT_START_IGNORE,
		 PROMPTCOLOUR,
		 '\001', RL_PROMPT_END_IGNORE,
		 PACKAGE ">",
		 '\001', RL_PROMPT_START_IGNORE,
		 DEFAULTCOLOUR,
		 '\001', RL_PROMPT_END_IGNORE );
	
	/* Clear the screen
	*/
	printf( "%s", DEFAULTCOLOUR );

	/* Initialise the calculation engine
	*/
	rCalc_engine_Initialise();
}

void rCalc_console_ProcessUserInput( void )
{
	while( 1 )
	{
		char *rawLine = readline( user_prompt );
		int quit_flag = 0;
		char *line;
		
		/* CTRL-D = exit
		*/
		if( !rawLine )
		{
			printf( "\n" );
			break;
		}

		/* Remove leading and trailing whitespace.  line and,
		** if there is anything left, add it to the history
		** list and execute it.
		*/
		line = stripLeadAndTrailSpace( rawLine );
		
		/* If there is anything left, add it to the history
		** and execute it.
		*/
		if( *line )
		{
			add_history( line );
			quit_flag = rCalc_engine_Execute( line );
		}
		free( rawLine );

		if( quit_flag ) break;
	}
}

void rCalc_console_Shutdown( void )
{
	/* Write the history.
	*/
	if( historyFile ) write_history( historyFile );
}

/*****************************************************************************/

#if HAVE_VPRINTF!=1
#error "This program relies on the vprintf() function."
#endif

/* Formatted printing.
*/
void Print( char *fmt, ... )
{
	va_list ap;

	va_start( ap, fmt );
	vprintf( fmt, ap );
	va_end( ap );

	fflush( stdout );
}
void Error( char *fmt, ... )
{
	va_list ap;

	/* Engine | Prefix for all error messages */
	printf( "%s%s ", ERRORCOLOUR, _("error:") );

	va_start( ap, fmt );
	vprintf( fmt, ap );
	va_end( ap );

	printf( "%s\n", DEFAULTCOLOUR );
	fflush( stdout );
}
void Debug( char *fmt, ... )
{
#ifdef PRINT_DEBUGGING_MESSAGES
	va_list ap;

	printf( "%s%s ", DEBUGCOLOUR, "debug:" );

	va_start( ap, fmt );
	vprintf( fmt, ap );
	va_end( ap );

	printf( "%s\n", DEFAULTCOLOUR );
	fflush( stdout );
#endif
}

/*** end of console.c ********************************************************/
